package com.fluentcommerce.util.test.executor;

import com.apollographql.apollo.api.Operation;
import com.apollographql.apollo.api.Query;
import com.fluentcommerce.util.dynamic.graphql.DynamicEntityQuery;
import com.fluentretail.api.v2.client.ReadOnlyFluentApiClient;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.workflow.RuleInstance;
import com.fluentretail.rubix.workflow.Workflow;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import static com.fluentcommerce.util.test.TestUtils.eventWithDefaults;
import static com.fluentcommerce.util.test.TestUtils.loadWorkflowFromFile;
import static com.fluentcommerce.util.test.TestUtils.validateEventAgainstProduces;

public class WorkflowExecutor {
    private final MockApiClient apiMock;
    private final ReadOnlyFluentApiClient api;
    private final Workflow workflow;
    private final Map<String, Consumer<RuleContextGenerator.RuleTestContext>> ruleMocks = new HashMap<>();
    private final Map<String, MockApiClient> apiMocks = new HashMap<>();
    private static final int NUMBER_OF_RULE_NAME_PATH = 3;

    public WorkflowExecutor(final Workflow workflow) {
        this.apiMock = new MockApiClient();
        // TODO: variant that actually makes calls against a live account instead of mocking everything (for debugging)
        this.api = apiMock.get();
        this.workflow = workflow;
    }

    /**
     * Supply an anonymous function to be called when the named rule is encountered in the workflow.
     *
     * @param name full name of the rule, including account and plugin namespace.
     * @param rule Consumer that will be called. Receives the context object.
     * @return this WorkflowExecutor to allow config chaining.
     */
    public WorkflowExecutor mockRule(final String name, final Consumer<RuleContextGenerator.RuleTestContext> rule) {
        ruleMocks.put(name, rule);
        return this;
    }

    /**
     * Uses the given legacy client instead of a stubbed one.
     *
     * @param legacyExecutor the legacy executor to use
     * @return this instance
     */
    public WorkflowExecutor withLegacyClient(final String ruleName,
                                             final com.fluentretail.api.client.ReadOnlyFluentApiClient legacyExecutor) {
        final MockApiClient apiClient = getApiMock(ruleName);
        apiClient.withLegacyClient(legacyExecutor);
        return this;
    }

    /**
     * Supply a function for ignore a bunch of rules
     *
     * @param names list of full name of the rule, including account and plugin namespace.
     * @return this WorkflowExecutor to allow config chaining.
     */
    public WorkflowExecutor ignoreRules(final String... names) {
        for (final String name : names) {
            ruleMocks.put(name, t -> {});
        }
        return this;
    }

    private RuleRepository.ExecutableRule getRule(final RuleInstance ruleInstance) {
        // assuming ruleInstance.getName() is "[[account.id]].plugin.rule" or "account.plugin.rule"
        String[] rulePaths = ruleInstance.getName().split("\\.");

        // In case if ruleInstance.getName() is not "<account>.<plugin>.<rule>" format
        if (rulePaths.length < NUMBER_OF_RULE_NAME_PATH) {
            return ruleMocks.containsKey(ruleInstance.getName())
                ? RuleRepository.MockExecutableRule.of(ruleMocks.get(ruleInstance.getName()))
                : RuleRepository.get(ruleInstance.getName());
        } else {
            rulePaths = new String[]{rulePaths[rulePaths.length - 2], rulePaths[rulePaths.length - 1]};
        }

        // check ruleMocks for "<account>.<plugin>.<rule>"
        if (ruleMocks.containsKey(ruleInstance.getName())) {
            return RuleRepository.MockExecutableRule.of(ruleMocks.get(ruleInstance.getName()));
        }

        // check ruleMocks for "<plugin>.<rule>"
        String ruleName = rulePaths[0] + rulePaths[1];
        if (ruleMocks.containsKey(ruleName)) {
            return RuleRepository.MockExecutableRule.of(ruleMocks.get(ruleName));
        }

        // check ruleMocks for "<rule>"
        ruleName = rulePaths[1];
        if (ruleMocks.containsKey(ruleName)) {
            return RuleRepository.MockExecutableRule.of(ruleMocks.get(ruleName));
        }

        return RuleRepository.get(ruleName);
    }

    /**
     * Execute the workflow and return the TestContext with any actions that were produced.
     * <p>
     * Note: this version runs the rule with a default Event with no attributes and
     * referencing an Order entity. If your rule requires specific event attributes and/or
     * entity information, use the `execute(Event eventOverrides)` variant instead.
     *
     * @return TestContext containing actions produced during execution.
     */
    public RuleContextGenerator execute() {
        return execute(Event.builder().build());
    }

    /**
     * Execute the workflow and return the TestContext with any actions that were produced.
     * <p>
     * Sample usage:
     *   `ex.execute(Event.builder().attributes(ImmutableMap.of("product", product)).build())`
     *
     * @param eventOverrides a (partial) Event that contains the attributes and/or entity
     *                       information required to execute this workflow. Any null fields will
     *                       be auto-populated with values referencing an Order entity.
     *                       See: `TestUtils.eventWithDefaults(Event event)` for details.
     *
     * @return TestContext containing actions produced during execution.
     */
    public RuleContextGenerator execute(final Event eventOverrides) {
        final Event event = eventWithDefaults(eventOverrides);
        final RuleContextGenerator context = RuleContextGenerator.of(event, api);
        do {
            final Event thisEvent = context.getEvent();
            workflow.getRulesets().stream().filter(rs -> rs.matches(thisEvent)).forEach(rs -> {
                rs.getRules().forEach(rule -> {
                    final RuleRepository.ExecutableRule thisRule = getRule(rule);
                    thisRule.run(context.mockApiClient(getApiMock(rule.getName())).forRule(rule));
                    validateEventAgainstProduces(thisRule, context, rule);
                });
            });
        } while(context.nextEvent());
        return context;
    }

    public static WorkflowExecutor of(final Workflow workflow) {
        return new WorkflowExecutor(workflow);
    }

    public static WorkflowExecutor of(final String workflowFile) {
        return new WorkflowExecutor(loadWorkflowFromFile(workflowFile));
    }

    public <D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D,T,V>> WorkflowExecutor mockNamedQuery(final String ruleName, final Class<Q> query, final String file, final String... moreFiles) {
        apiMocks.put(ruleName, apiMocks.getOrDefault(ruleName, new MockApiClient()).mockNamedQuery(query, file, moreFiles));
        return this;
    }

    public <D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D,T,V>> WorkflowExecutor mockNamedQuery(final String ruleName, final Class<Q> query, final D data, final D... moreData) {
        apiMocks.put(ruleName, apiMocks.getOrDefault(ruleName, new MockApiClient()).mockNamedQuery(query, data, moreData));
        return this;
    }

    public WorkflowExecutor mockDynamic(final String ruleName, final String file) {
        apiMocks.put(ruleName, new MockApiClient().mockNamedQuery(DynamicEntityQuery.class, file));
        return this;
    }

    public MockApiClient getApiMock() {
        return this.apiMock;
    }

    private MockApiClient getApiMock(final String ruleName) {
        return apiMocks.getOrDefault(ruleName, apiMock);
    }
}
