package com.fluentcommerce.util.test.executor;

import com.apollographql.apollo.api.Mutation;
import com.fluentcommerce.util.test.ApolloUtils;
import com.fluentretail.api.model.attribute.Attribute;
import com.fluentretail.rubix.action.Action;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.action.ActionFactory;
import com.fluentretail.rubix.workflow.ServiceManager;
import lombok.AllArgsConstructor;
import lombok.Value;

import java.util.ArrayList;
import java.util.List;

public class TestActions {

    @Value(staticConstructor = "of")
    public static class TestActionFactory implements ActionFactory {

        RuleContextGenerator.RuleTestContext.RuleContext ctx;

        @Override public void mutation(Mutation mutation) {
            ctx.pushAction(new MutateAction(mutation));
        }

        @Override
        public void mutation(Mutation mutation, String s) {
            mutation(mutation);
        }

        @Override public void sendEvent(Event event) {
            ctx.pushEvent(event);
            ctx.pushAction(new SendEventAction(event));
        }

        @Override
        public void log(String message, String detailedMessage, List<Attribute> attributes) {
            // Rubix tries to append to the log attributes so check for mutability
            try {
                attributes.add(Attribute.of("mutability_test", "success"));
            } catch(Exception e) {
                throw new RuntimeException("Log event only supports a mutable list of attributes");
            }

            ctx.pushAction(new LogAction(ctx.getEvent(), message, detailedMessage, attributes));
        }

        @Override public void postWebhook(String url, Event event) {
            ctx.pushAction(WebhookAction.get(url, event));
        }

    }

    private static class BaseAction implements Action {
        @Override
        public List<com.fluentretail.api.model.event.Event> fire(ServiceManager serviceManager) {
            System.out.println("Fired action: " + this.getClass());
            return new ArrayList<>();
        }
    }

    @Value
    @AllArgsConstructor
    public static class MutateAction extends BaseAction {

        public Mutation mutation;

        /**
         * Retrieve the "input" variable as a specified type.
         *
         * This is often the only variable on a mutation, so is useful
         * for asserting outcomes in executor-based tests.
         *
         * @param inputType Class of the expected response type
         * @param <T> Class of the expected response type
         * @return the "input" variable converted to T
         */
        public <T> T getMutationInputAs(Class<T> inputType) {
            return ApolloUtils.getMutationVariable(mutation, "input", inputType);
        }
    }

    @Value
    @AllArgsConstructor
    public static class SendEventAction extends BaseAction {

        Event event;

        public boolean isFutureDated() {
            return event.getScheduledOn() != null;
        }
    }

    @Value
    @AllArgsConstructor
    public static class LogAction extends BaseAction {
        Event sourceEvent;
        String message;
        String detailedMessage;
        List<Attribute> attributes;
    }

    @Value
    @AllArgsConstructor
    public static class WebhookAction extends BaseAction {

        String webhookEndpoint;
        Event event;

        public static WebhookAction get(String webhookEndpoint, Event event) {
            return new WebhookAction(webhookEndpoint, event);
        }
    }
}
