package com.fluentcommerce.util.test.executor;

import com.fluentretail.rubix.exceptions.RuleNotFoundException;
import com.fluentretail.rubix.rule.BaseRule;
import com.fluentretail.rubix.rule.meta.EventAttribute;
import com.fluentretail.rubix.rule.meta.ParamAttributeType;
import com.fluentretail.rubix.rule.meta.ParamEnum;
import com.fluentretail.rubix.rule.meta.ParamInteger;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.ParamTimePeriod;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.rubix.workflow.RuleInstance;
import com.google.common.collect.ImmutableMap;
import lombok.Value;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.test.TestUtils.gatherRules;
import static com.fluentcommerce.util.test.TestUtils.getRuleName;

public class RuleRepository {
    static final Map<String, Rule> rules;
    static {
        try {
            ImmutableMap.Builder<String, Rule> builder = ImmutableMap.builder();
            gatherRules().forEach((c) -> builder.put(c.getAnnotation(RuleInfo.class).name(), instantiateRule(c)));
            rules = builder.build();
        } catch (IllegalArgumentException e) {
            throw new RuntimeException("Multiple rules have the same name! Check your @RuleInfo details.", e);
        }
        System.out.println("TestExecutor gathered rules: " + rules);
    }

    private static <T extends BaseRule> T instantiateRule(Class<T> clazz) {
        try {
            return clazz.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            System.out.println("Failed to instantiate Rule class " + clazz);
            e.printStackTrace();
        }
        return null;
    }

    public static ExecutableRule get(final RuleInstance inst) {
        return get(inst.getName());
    }

    public static ExecutableRule get(final String name) {
        final String ruleName = getRuleName(name);
        final Rule rule = rules.get(ruleName);
        if (rule != null) {
            return ExecutableRuleImpl.of(rule);
        }
        final List<String> v2rules = rules.values().stream()
                .map(r -> r.getClass().getSimpleName())
                .collect(Collectors.toList());
        throw new RuleNotFoundException(name, v2rules);
    }

    public interface ExecutableRule {
        void run(RuleContextGenerator.RuleTestContext context);
        RuleInfo info();
        ImmutableMap<String, Annotation> params();
        ImmutableMap<String, EventAttribute> attributes();
    }

    @Value(staticConstructor = "of")
    public static class ExecutableRuleImpl implements ExecutableRule {
        Rule rule;

        @Override
        public void run(final RuleContextGenerator.RuleTestContext context) {
            rule.run(context.context());
        }

        @Override
        public RuleInfo info() {
            return rule.getClass().getAnnotation(RuleInfo.class);
        }

        @Override
        public ImmutableMap<String, Annotation> params() {
            final ImmutableMap.Builder<String, Annotation> builder = ImmutableMap.builder();
            for (ParamString anno: rule.getClass().getAnnotationsByType(ParamString.class)) {
                builder.put(anno.name(), anno);
            }
            for (ParamInteger anno: rule.getClass().getAnnotationsByType(ParamInteger.class)) {
                builder.put(anno.name(), anno);
            }
            for (ParamEnum anno: rule.getClass().getAnnotationsByType(ParamEnum.class)) {
                builder.put(anno.name(), anno);
            }
            for (ParamTimePeriod anno: rule.getClass().getAnnotationsByType(ParamTimePeriod.class)) {
                builder.put(anno.name(), anno);
            }
            for (ParamAttributeType anno: rule.getClass().getAnnotationsByType(ParamAttributeType.class)) {
                builder.put(anno.name(), anno);
            }
            return builder.build();
        }

        @Override
        public ImmutableMap<String, EventAttribute> attributes() {
            final ImmutableMap.Builder<String, EventAttribute> builder = ImmutableMap.builder();
            for(EventAttribute anno: rule.getClass().getAnnotationsByType(EventAttribute.class)) {
                builder.put(anno.name(), anno);
            }
            return builder.build();
        }
    }

    @RuleInfo(name = "MockRule", description = "A mock rule")
    public static class EmptyMockRule {}

    @Value(staticConstructor = "of")
    public static class MockExecutableRule implements ExecutableRule {
        Consumer<RuleContextGenerator.RuleTestContext> mock;

        @Override
        public void run(RuleContextGenerator.RuleTestContext context) {
            mock.accept(context);
        }

        @Override
        public RuleInfo info() {
            return EmptyMockRule.class.getAnnotation(RuleInfo.class);
        }

        @Override
        public ImmutableMap<String, Annotation> params() {
            return ImmutableMap.of();
        }

        @Override
        public ImmutableMap<String, EventAttribute> attributes() {
            return ImmutableMap.of();
        }
    }
}
