package com.fluentcommerce.util.test.executor;

import com.apollographql.apollo.api.Operation;
import com.apollographql.apollo.api.Query;
import com.fluentcommerce.graphql.queries.introspection.IntrospectionQuery;
import com.fluentretail.api.client.ReadOnlyFluentApiClient;
import com.fluentcommerce.util.dynamic.graphql.DynamicEntityQuery;
import com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.rubix.workflow.RuleInstance;

import java.util.Map;

import static com.fluentcommerce.util.test.TestUtils.validateEventAgainstProduces;

/**
 * The RuleExecutor allows unit tests to run a rule in a similar way to Rubix.
 *
 * It allows:
 * - control over inputs with sensible defaults
 * - simple mocking of GraphQL query responses
 * - access to any actions produced during execution (for assertions)
 */
public class RuleExecutor {

    private MockApiClient api;

    private final RuleInstance rule;

    private RuleExecutor(final RuleInstance rule) {
        this.api = new MockApiClient();
        this.rule = rule;
    }

    public <D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D,T,V>> RuleExecutor mockNamedQuery(final Class<Q> query, final String file, final String... moreFiles) {
        api.mockNamedQuery(query, file, moreFiles);
        return this;
    }

    public <D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D,T,V>> RuleExecutor mockNamedQuery(final Class<Q> query, final D data, final D... moreData) {
        api.mockNamedQuery(query, data, moreData);
        return this;
    }

    public RuleExecutor mockDynamic(final String file) {
        api.mockNamedQuery(DynamicEntityQuery.class, file);
        return this;
    }

    /**
     * Execute the rule with a default Event and return the TestContext with any actions that were produced.
     *
     * Note: this version runs the rule with a default Event with no attributes and
     * referencing an Order entity. If your rule requires specific event attributes and/or
     * entity information, use the `execute(Event eventOverrides)` variant instead.
     *
     * @return TestContext containing actions produced during execution.
     */
    public RuleContextGenerator execute() {
        return execute(Event.builder().build());
    }

    /**
     * Execute the rule with the particular Event and return the TestContext with any actions that were produced.
     *
     * Sample usage:
     *   `ex.execute(Event.builder().attributes(ImmutableMap.of("product", product)).build())`
     *
     * @param eventOverrides a (partial) Event that contains the attributes and/or entity
     *                       information required to execute this rule. Any null fields will
     *                       be auto-populated with values referencing an Order entity.
     *                       See: `TestUtils.eventWithDefaults(Event event)` for details.
     *
     * @return TestContext containing actions produced during execution.
     */
    public RuleContextGenerator execute(final Event eventOverrides) {
        // first mock the introspection query dynamic queries use to validate
        if(!GraphQLIntrospectionUtils.isSchemaLoaded()) {
            QueryMock.of(IntrospectionQuery.class, "graphql/introspection/schema.json").mockFor(api.get());
        }
        RuleRepository.ExecutableRule thisRule = RuleRepository.get(rule);
        RuleContextGenerator context = RuleContextGenerator.of(eventOverrides, api.get());
        thisRule.run(context.forRule(rule));
        // validate events against @produces annotation
        validateEventAgainstProduces(thisRule, context, rule);
        return context;
    }

    /**
     * Create a new API mock, effectively clearing out any previously mocked queries.
     *
     * @return this instance
     */
    public RuleExecutor resetApi() {
        this.api = new MockApiClient();
        return this;
    }

    /**
     * Replace this executors API mock with your own.
     *
     * This allows you to re-use the same mocks across different tests if desired.
     *
     * @param api instance of MockApiClient
     * @return this instance
     */
    public RuleExecutor replaceApi(final MockApiClient api) {
        this.api = api;
        return this;
    }

    /**
     * Uses the given legacy client instead of a stubbed one.
     *
     * @param legacyExecutor the legacy executor to use
     * @return this instance
     */
    public RuleExecutor withLegacyClient(final ReadOnlyFluentApiClient legacyExecutor) {
        this.api.withLegacyClient(legacyExecutor);
        return this;
    }

    /**
     * Create a RuleExecutor for the specified Rule by name.
     *
     * @param ruleName name of the rule to test, as per the @RuleInfo annotation
     * @param props Map of Rule configuration properties
     * @return a new RuleExecutor ready to execute
     */
    public static RuleExecutor of(final String ruleName, final Map<String, Object> props) {
        return new RuleExecutor(RuleInstance.builder().name(ruleName).props(props).build());
    }

    /**
     * Create a RuleExecutor for the specified Rule by class.
     *
     * @param ruleClass class of the rule to test
     * @param props Map of Rule configuration properties
     * @return a new RuleExecutor ready to execute
     */
    public static RuleExecutor of(final Class<? extends Rule> ruleClass, final Map<String, Object> props) {
        return new RuleExecutor(
                RuleInstance.builder()
                        .name(ruleClass.getAnnotation(RuleInfo.class).name())
                        .props(props)
                    .build());
    }
}
