package com.fluentcommerce.util.test.executor;

import com.fluentretail.api.v2.client.ReadOnlyFluentApiClient;
import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.rubix.action.Action;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.exceptions.RuleNotFoundException;
import com.fluentretail.rubix.util.ValueConverter;
import com.fluentretail.rubix.v2.action.ActionFactory;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.workflow.RuleInstance;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.test.TestUtils.entityFromEvent;
import static com.fluentcommerce.util.test.TestUtils.eventWithDefaults;

/**
 * Created by ben on 11/10/16.
 */
@Slf4j
public class RuleContextGenerator {

    // List of actions that have been produced
    private final List<Action> actions = new ArrayList<>();

    // pending events (for workflow-style tests)
    private final LinkedList<Event> eventQueue = new LinkedList<>();

    // api client (usually mocked)
    private ReadOnlyFluentApiClient api;

    // entity being operated on
    private final Entity primaryEntity;

    private RuleContextGenerator(final Event event, final ReadOnlyFluentApiClient api, final Entity primaryEntity) {
        eventQueue.add(event);
        this.api = api;
        this.primaryEntity = primaryEntity;
    }

    /**
     * Replace this executors API mock with your own.
     * <p>
     * This allows you to re-use the same mocks across different tests if desired.
     *
     * @param api instance of MockApiClient
     * @return this instance
     */
    public RuleContextGenerator mockApiClient(final MockApiClient api) {
        this.api = api.get();
        return this;
    }

    public RuleContextGenerator pushEvent(final Event event) {
        eventQueue.push(event);
        return this;
    }

    /**
     * Retrieve the actions that were produced during execution.
     * @return all of the actions.
     */
    public List<Action> getActions() {
        return actions;
    }

    /**
     * Retrieve the actions that were produced during execution, filtered by type.
     * @return filtered list of typed actions.
     */
    public <T extends Action> List<T> getActionsOfType(final Class<T> type) {
        return (List<T>) actions.stream()
                .filter(a -> type.isAssignableFrom(a.getClass()))
                .collect(Collectors.toList());
    }

    /**
     * Retrieve the first action of a given type that was produced during execution.
     * @return null or one typed action.
     */
    public <T extends Action> T getFirstActionOfType(final Class<T> type) {
        final List<T> typedActions = getActionsOfType(type);
        return !typedActions.isEmpty() ? typedActions.get(0) : null;
    }

    /**
     * Retrieve the last action of a given type that was produced during execution.
     * @return null or one typed action.
     */
    public <T extends Action> T getLastActionOfType(final Class<T> type) {
        final List<T> typedActions = getActionsOfType(type);
        return !typedActions.isEmpty() ? typedActions.get(typedActions.size() - 1) : null;
    }

    public boolean hasNextEvent() {
        return eventQueue.size() > 1;
    }

    public boolean nextEvent() {
        if (!hasNextEvent()) {
            return false;
        }
        eventQueue.poll();
        return true;
    }

    public Event getEvent() {
        return eventQueue.peek();
    }

    public RuleTestContext forRule(final RuleInstance rule) {
        return new RuleTestContext(rule);
    }

    public RuleTestContext noRule() {
        return new RuleTestContext(null);
    }

    @Value
    @AllArgsConstructor
    public class RuleTestContext {
        RuleInstance rule;

        public RuleContext context() {
            return new RuleContext();
        }

        public final class RuleContext implements Context {
            @Override
            public String getProp(final String name) {
                this.validateProp(RuleTestContext.this.rule.getName(), "String", name);
                return this.getProp(name, String.class);
            }

            @Override
            public <T> T getProp(final String name, final Class<T> type) {
                this.validateProp(RuleTestContext.this.rule.getName(), type.getSimpleName(), name);
                return RuleTestContext.this.rule.getProps() != null && RuleTestContext.this.rule.getProps().containsKey(name) ? ValueConverter.convert(RuleTestContext.this.rule.getProps().get(name), type) : ValueConverter.convert(null, type);
            }

            @Override
            public <T> List<T> getPropList(final String name, final Class<T> itemType) {
                this.validateProp(RuleTestContext.this.rule.getName(), "List of " + itemType.getSimpleName(), name);
                return RuleTestContext.this.rule.getProps() != null && RuleTestContext.this.rule.getProps().containsKey(name) ? ValueConverter.convertList(RuleTestContext.this.rule.getProps().get(name), itemType) : ValueConverter.convertList(null, itemType);
            }

            private void validateProp(final String ruleName, final String type, final String prop) {
                try {
                    final Annotation annotation = RuleRepository.get(RuleTestContext.this.rule).params().get(prop);
                    if (annotation == null) {
                        throw new IllegalArgumentException(String.format("Rule [%s] tried to use an undocumented %s prop: [%s]", ruleName, type, prop));
                    }
                } catch (RuleNotFoundException var5) {
                    if (RuleContextGenerator.log.isDebugEnabled()) {
                        RuleContextGenerator.log.debug("Rule '{}' with prop '{}' of type '{}' not found", ruleName, prop, type);
                    }
                }
            }

            @Override
            public Event getEvent() {
                return eventQueue.peek();
            }

            @Override
            public Entity getEntity() {
                return primaryEntity;
            }

            @Override
            public ReadOnlyFluentApiClient api() {
                return api;
            }

            @Override
            public ActionFactory action() {
                return TestActions.TestActionFactory.of(this);
            }

            @Override
            public Context pushAction(final Action action) {
                actions.add(action);
                return this;
            }

            public Context pushEvent(final Event event) {
                eventQueue.add(event);
                return this;
            }
        }
    }

    public static RuleContextGenerator of(final ReadOnlyFluentApiClient api) {
        return of(Event.builder().build(), api);
    }

    public static RuleContextGenerator of(final Event eventOverrides, final ReadOnlyFluentApiClient api) {
        Event event = eventWithDefaults(eventOverrides);
        Entity primaryEntity = entityFromEvent(event);
        return new RuleContextGenerator(event, api, primaryEntity);
    }
}
