package com.fluentcommerce.util.test.executor;

import com.apollographql.apollo.CustomTypeAdapter;
import com.apollographql.apollo.api.Operation;
import com.apollographql.apollo.api.Query;
import com.apollographql.apollo.api.ScalarType;
import com.apollographql.apollo.internal.cache.normalized.ResponseNormalizer;
import com.apollographql.apollo.internal.response.OperationResponseParser;
import com.apollographql.apollo.internal.response.ScalarTypeAdapters;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.databind.util.ISO8601DateFormat;
import com.fluentcommerce.graphql.queries.introspection.IntrospectionQuery;
import com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes;
import com.fluentcommerce.util.dynamic.graphql.DynamicEntityQuery;
import com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils;
import com.fluentretail.api.client.ObjectMapperFactory;
import com.fluentretail.api.v2.client.ReadOnlyFluentApiClient;
import com.fluentretail.graphql.type.CustomType;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import lombok.AllArgsConstructor;
import lombok.Value;
import okio.BufferedSource;
import okio.Okio;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.text.ParseException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.dynamic.JsonUtils.streamToNode;
import static org.mockito.ArgumentMatchers.isA;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

//@Value(staticConstructor = "of")
@Value
@AllArgsConstructor
public class QueryFileMock<D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D, T, V>> implements QueryMock {
    private static final Map<Class<?>, Object> DEFAULTS = ImmutableMap.of(
            String.class, "",
            Integer.class, 1,
            List.class, ImmutableList.of(),
            boolean.class, true
    );

    public static final Map<ScalarType, CustomTypeAdapter> ADAPTERS = ImmutableMap.of(
            CustomType.DATETIME, new DateAdapter(),
            CustomType.JSON, new JsonAdapter()
    );

    Class<Q> query;
    String file;
    List<String> moreFiles;

    private static Map<String, Operation.Data> mockDataCache = new HashMap<>();

    @Override
    public void mockFor(final ReadOnlyFluentApiClient mockClient) {
        final Operation.Data first = toOperationData(query, file, mockClient);
        final Operation.Data[] others = moreFiles.stream()
                .map(file -> toOperationData(query, file, mockClient))
                .toArray(Operation.Data[]::new);

        when(mockClient.query(isA(query))).thenReturn(first, others);
    }

    private Operation.Data toOperationData(final Class<Q> clazz, final String filename, final ReadOnlyFluentApiClient client) {
        final String key = clazz.getName() + filename;
        if (mockDataCache.containsKey(key)) {
            return mockDataCache.get(key);
        }

        if (clazz == DynamicEntityQuery.class) {
            mockDataCache.put(key, loadDynamicQueryResultFromFile(client, filename));
            return mockDataCache.get(key);
        }

        mockDataCache.put(key, (Operation.Data) loadQueryResultFromFile(clazz, filename));
        return mockDataCache.get(key);
    }

    public static <D extends Operation.Data, T, V extends Operation.Variables, Q extends Query<D, T, V>> T loadQueryResultFromFile(final Class<Q> clazz, final String filename) {
        Q query;

        try {
            final Constructor[] constructor = clazz.getDeclaredConstructors();
            final Constructor first = constructor[0];
            for (final Class parameterType : first.getParameterTypes()) {
                DEFAULTS.get(parameterType);
            }

            final List<Object> params = ImmutableList.copyOf(first.getParameterTypes()).stream()
                    .map(DEFAULTS::get).collect(Collectors.toList());

            query = (Q) constructor[0].newInstance(params.toArray());

            final OperationResponseParser<D, T> parser =
                    new OperationResponseParser<>(query, query.responseFieldMapper(),
                            new ScalarTypeAdapters(ADAPTERS), mock(ResponseNormalizer.class)
                    );

            final ClassLoader classloader = Thread.currentThread().getContextClassLoader();
            final InputStream is = classloader.getResourceAsStream(filename);
            final BufferedSource bufferedSource = Okio.buffer(Okio.source(is));
            return parser.parse(bufferedSource).data();

        } catch (IllegalAccessException | InstantiationException | InvocationTargetException | IOException e) {
            throw new RuntimeException("GQL mock failed for file: " + filename, e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException("GQL mock response file doesn't appear to exist: " + filename, e);
        }

    }

    public static DynamicDataTypes.QueryDynamicData loadDynamicQueryResultFromFile(final ReadOnlyFluentApiClient mockClient, final String filename) {
        // first mock the introspection query dynamic queries use to validate
        if (!GraphQLIntrospectionUtils.isSchemaLoaded()) {
            QueryMock.of(IntrospectionQuery.class, "graphql/introspection/schema.json").mockFor(mockClient);
        }

        final ClassLoader classloader = Thread.currentThread().getContextClassLoader();
        final InputStream is = classloader.getResourceAsStream(filename);
        return new DynamicDataTypes.QueryDynamicData((ObjectNode) streamToNode(is));
    }

    // static mock functions below

    public static final class JsonAdapter implements CustomTypeAdapter<Object> {
        @Nonnull
        @Override
        public Object decode(@Nonnull final String source) {
            Object res;
            try {
                res = ObjectMapperFactory.get().readValue(source, JsonNode.class);
            } catch (IOException e) {
                e.printStackTrace();
                res = source;
            }

            if (res instanceof TextNode) {
                return ((TextNode) res).asText();
            }

            return res;
        }

        @Nonnull
        @Override
        public String encode(@Nonnull final Object value) {
            String strValue;
            try {
                strValue = ObjectMapperFactory.get().writeValueAsString(value);
            } catch (JsonProcessingException e) {
                e.printStackTrace();
                strValue = value.toString();
            }
            return strValue;
        }
    }


    public static final class DateAdapter implements CustomTypeAdapter<Date> {
        final ISO8601DateFormat df = new ISO8601DateFormat();

        @Override
        @Nonnull
        public Date decode(@Nonnull final String value) {
            String strValue = null;
            try {
                try {
                    strValue = ObjectMapperFactory.get().readValue(value, String.class);
                } catch (IOException ignored) {
                }

                return df.parse(strValue);
            } catch (ParseException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        @Nonnull
        public String encode(@Nonnull final Date value) {
            return "\"" + df.format(value) + "\"";
        }
    }
}
