package com.fluentcommerce.util.test;

import com.fluentcommerce.util.test.executor.RuleContextGenerator;
import com.fluentcommerce.util.test.executor.RuleRepository;
import com.fluentcommerce.util.test.executor.TestActions;
import com.fluentretail.api.model.RubixEntity;
import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.EventInfo;
import com.fluentretail.rubix.rule.meta.EventInfoVariables;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.rule.Rule;
import com.fluentretail.rubix.workflow.RuleInstance;
import com.fluentretail.rubix.workflow.Workflow;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.reflections.Reflections;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.dynamic.JsonUtils.anyToPojo;
import static com.fluentcommerce.util.dynamic.JsonUtils.streamToNode;

@Slf4j
public class TestUtils {

    /**
     * Scan the project and return the class object for all rules (except legacy).
     * <p>
     * This is used to build the RuleRepository before test executions, but can also be useful
     * for unit tests on rule metadata (enforcing naming standards etc).
     *
     * @return all classes extending the Rule interface in the current project.
     */
    public static Set<Class<? extends Rule>> gatherRules() {
        final Reflections reflections = new Reflections("");
        return reflections.getSubTypesOf(Rule.class).stream().filter(c -> c.isAnnotationPresent(RuleInfo.class)).collect(Collectors.toSet());
    }

    /**
     * Create an event with default values for unit tests.
     *
     * @return event with default values.
     */
    public static Event eventWithDefaults() {
        return eventWithDefaults(Event.builder().build());
    }

    /**
     * Create an event with some sensible default values for unit tests.
     * <p>
     * Any fields can be overwritten by passing in an event with that value.
     *
     * @param event field overrides
     * @return event with defaults wherever the inbound event was missing values.
     */
    public static Event eventWithDefaults(final Event event) {
        return event.toBuilder()
                .name((event.getName() == null) ? "Single Rule Test" : event.getName())
                .accountId((event.getAccountId() == null) ? "FLUENT" : event.getAccountId())
                .retailerId((event.getRetailerId() == null) ? "1" : event.getRetailerId())
                .rootEntityType((event.getRootEntityType() == null) ? "ORDER" : event.getRootEntityType())
                .rootEntityId((event.getRootEntityId() == null) ? "100" : event.getRootEntityId())
                .rootEntityRef((event.getRootEntityRef() == null) ? "REF-100" : event.getRootEntityRef())
                .entityType((event.getEntityType() == null) ? "ORDER" : event.getEntityType())
                .entityId((event.getEntityId() == null) ? "100" : event.getEntityId())
                .entityRef((event.getEntityRef() == null) ? "REF-100" : event.getEntityRef())
                .entitySubtype((event.getEntitySubtype() == null) ? "HD" : event.getEntitySubtype())
                .entityStatus((event.getEntityStatus() == null) ? "CREATED" : event.getEntityStatus())
                .build();
    }

    /**
     * Create a RubixEntity to match the event for use in unit tests.
     *
     * @param event fully-qualified event to pull entity data from.
     * @return a RubixEntity with all fields needed for testing Orchestration.
     */
    public static Entity entityFromEvent(final Event event) {
        return RubixEntity.builder()
                .id(event.getEntityId())
                .ref(event.getEntityRef())
                .entityType(event.getEntityType())
                .type(event.getEntitySubtype())
                .status(event.getEntityStatus())
                .flexType(event.getFlexType())
                .flexVersion(1)
                .build();
    }

    /**
     * Load a workflow from a json file.
     *
     * @param filename reference to the file relative to the resources folder.
     * @return a Rubix workflow instance
     */
    public static Workflow loadWorkflowFromFile(final String filename) {
        final ClassLoader classloader = Thread.currentThread().getContextClassLoader();
        try (InputStream is = classloader.getResourceAsStream(filename)) {
            return anyToPojo(streamToNode(is), Workflow.class);
        } catch (IOException ex) {
            log.error("Load a workflow from a json file error: ", ex);
        }
        return null;
    }

    /**
     * Get a rule name from a full namespace name.
     *
     * @param fullNameSpaceRuleName a full namespace rule name.
     * @return a rule name
     */
    public static String getRuleName(final String fullNameSpaceRuleName) {
        // assuming ruleInstance.getName() is "<account>.<plugin>.<rule>"
        final String[] rulePaths = fullNameSpaceRuleName.split("\\.");
        if (rulePaths.length > 1) {
            return rulePaths[rulePaths.length - 1];
        }
        return fullNameSpaceRuleName;
    }

    /**
     * Validate events against @RuleInfo.produces annotation
     *
     * @param executableRule       this executable rule
     * @param ruleContextGenerator TestContext containing actions produced during execution.
     * @param ruleInstance         specific RuleInstances to populate rule name & props
     * @throws IllegalArgumentException if event not validate against @RuleInfo.produces annotation
     */
    public static void validateEventAgainstProduces(final RuleRepository.ExecutableRule executableRule,
                                                    final RuleContextGenerator ruleContextGenerator,
                                                    final RuleInstance ruleInstance) {
        final RuleInfo ruleInfo = executableRule.info();
        for (final TestActions.SendEventAction eventAction : ruleContextGenerator.getActionsOfType(TestActions.SendEventAction.class)) {
            // ignore validate with MockExecutableRule (rule plugin outside the project)
            if (executableRule instanceof RuleRepository.MockExecutableRule) {
                continue;
            }
            if (Arrays.stream(ruleInfo.produces()).noneMatch(eventInfo ->
                validateEventAgainstProduces(ruleContextGenerator.forRule(ruleInstance), eventInfo, eventAction.getEvent()))) {
                throw new IllegalArgumentException(String.format("Rule '%s' produced an event it hasn't declared in the `@RuleInfo.produces` annotation: %s", ruleInstance.getName(), eventAction.getEvent()));
            }
        }
    }

    private static String renderFromContext(final RuleContextGenerator.RuleTestContext context, final String value) {
        return replace(value, Pattern.compile("\\{(.*?)}"), m -> {
            final String sub = m.group(1);
            if (sub.startsWith("event.")) {
                switch (sub.substring(6)) {
                    case "entityType": return context.context().getEntity().getEntityType();
                    case "entitySubtype": return context.context().getEntity().getType();
                    case "status": return context.context().getEntity().getStatus();
                }
            } else {
                final String prop = context.context().getProp(sub);
                return prop != null ? prop : "";
            }
            throw new IllegalArgumentException(String.format("No known substitution for `@RuleInfo.produces` template value '%s'", sub));
        });
    }

    private static String replace(final String input, final Pattern regex, final Function<Matcher, String> callback) {
        final StringBuffer resultString = new StringBuffer();
        final Matcher regexMatcher = regex.matcher(input);
        while (regexMatcher.find()) {
            regexMatcher.appendReplacement(resultString, callback.apply(regexMatcher));
        }
        regexMatcher.appendTail(resultString);
        return resultString.toString();
    }

    private static boolean validateEventAgainstProduces(final RuleContextGenerator.RuleTestContext context,
                                                        final EventInfo info,
                                                        final Event event) {
        if (!renderFromContext(context, info.eventName()).equalsIgnoreCase(event.getName())) {
            log.error(String.format("Name not matching: %s -vs- %s", info.eventName(), event.getName()));
            return false;
        }
        if (!renderFromContext(context, producesDefault(info.entityType(), EventInfoVariables.EVENT_TYPE, context.context().getEntity().getEntityType())).equalsIgnoreCase(event.getEntityType())) {
            log.error(String.format("EntityType not matching: %s -vs- %s", info.entityType(), event.getEntityType()));
            return false;
        }
        if (!renderFromContext(context, producesDefault(info.entitySubtype(), EventInfoVariables.EVENT_SUBTYPE, context.context().getEntity().getType())).equalsIgnoreCase(event.getEntitySubtype())) {
            log.error(String.format("EntitySubtype not matching: %s -vs- %s", info.entitySubtype(), event.getEntitySubtype()));
            return false;
        }
        if (StringUtils.isNotBlank(info.status()) && !info.status().equals(EventInfoVariables.EVENT_STATUS) && !renderFromContext(context, info.status()).equalsIgnoreCase(event.getEntityStatus())) {
            log.error(String.format("EntityStatus not matching: %s -vs- %s", info.status(), event.getEntityStatus()));
            return false;
        }
        return true;
    }

    private static String producesDefault(String value, String placeholder, String defaultValue) {
        if (StringUtils.isBlank(value) || value.equals(placeholder)) {
            return defaultValue;
        }
        return value;
    }
}
