package com.fluentcommerce.util.test;

import com.apollographql.apollo.api.Mutation;
import com.apollographql.apollo.internal.json.InputFieldJsonWriter;
import com.apollographql.apollo.internal.json.JsonWriter;
import com.apollographql.apollo.internal.response.ScalarTypeAdapters;
import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.test.executor.QueryFileMock;
import okio.Okio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import static java.nio.charset.StandardCharsets.UTF_8;

public class ApolloUtils {

    /**
     * Retrieve a single variable from an Apollo mutation as a Pojo for test assertions.
     * <p>
     * See `getMutationVariables(Mutation mutation)` for details.
     *
     * @param mutation an Apollo Mutation
     * @param name name of the variable to cast and return
     * @param type expected POJO class of the response
     * @param <T> expected POJO class of the response
     * @return the variables object of the mutation as a T
     */
    public static <T> T getMutationVariable(final Mutation mutation, final String name, final Class<T> type) {
        JsonNode node = getMutationVariables(mutation);
        return JsonUtils.anyToPojo(node.get(name), type);
    }

    /**
     * Retrieve the variables from an Apollo mutation as a Pojo for test assertions.
     * <p>
     * See `getMutationVariables(Mutation mutation)` for details.
     *
     * @param mutation an Apollo Mutation
     * @param type expected POJO class of the response
     * @param <T> expected POJO class of the response
     * @return the variables object of the mutation as a T
     */
    public static <T> T getMutationVariables(final Mutation mutation, final Class<T> type) {
        return JsonUtils.anyToPojo(getMutationVariables(mutation), type);
    }

    /**
     * Retrieve the variables from an Apollo mutation as a JsonNode for test assertions.
     * <p>
     * This is _not_ performant and is intended only for unit test usage. It's working
     * around the fact that Apollo's own `mutation.variables().valueMap()` is an
     * unimplemented stub method.
     *
     * @param mutation an Apollo Mutation
     * @return the variables object of the mutation as a JsonNode
     */
    public static JsonNode getMutationVariables(final Mutation mutation) {
        try {
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            JsonWriter jsonWriter = JsonWriter.of(Okio.buffer(Okio.sink(bytes)));
            jsonWriter.setSerializeNulls(true);
            jsonWriter.beginObject();

            InputFieldJsonWriter fieldWriter = new InputFieldJsonWriter(
                    jsonWriter, new ScalarTypeAdapters(QueryFileMock.ADAPTERS));

            mutation.variables().marshaller().marshal(fieldWriter);

            jsonWriter.endObject();
            jsonWriter.close();

            return JsonUtils.stringToNode(new String(bytes.toByteArray(), UTF_8));
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }
}
