/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fluentcommerce.util.sourcing.SourcingCriteriaSettingGenerator;
import org.junit.jupiter.api.Test;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaTypeRegistry.generateSourcingCriteriaSetting;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

class SourcingCriteriaTypeRegistryTest {

    private static final ObjectMapper objectMapper = new ObjectMapper();

    static {
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
    }

    @Test
    void should_generate_json_for_criteria() throws IOException {
        List<Object> criteriaConfigs = new ArrayList<>();
        for (String typeName : SourcingCriteriaTypeRegistry.getTypeNameToClassMap().keySet()) {
            SourcingCriterion criterion = SourcingCriteriaTypeRegistry.getSourcingCriterion(typeName);
            criteriaConfigs.add(objectMapper.readTree(criterion.toJson().toString()));
        }
        String expectedContext;
        try (InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("criteria/SourcingCriteriaSetting.json");
             BufferedReader br = new BufferedReader(new InputStreamReader(is))) {
            expectedContext = br.lines().collect(Collectors.joining("\n"));
        }

        assertEquals(expectedContext, objectMapper.writeValueAsString(criteriaConfigs));
    }

    @Test
    void should_generate_json_file_for_criteria() throws IOException, URISyntaxException {
        Path out = resolveOutPath();
        Files.deleteIfExists(out);

        generateSourcingCriteriaSetting();
        assertTrue(Files.exists(out), "expected generated file at: " + out);

        byte[] bytes = Files.readAllBytes(out);
        String content = new String(bytes, StandardCharsets.UTF_8);
        assertNotNull(content);
        assertFalse(content.isEmpty());

        ObjectMapper mapper = new ObjectMapper();
        JsonNode root = mapper.readTree(content);
        assertTrue(root.isArray());
        assertTrue(root.size() > 0);

        JsonNode first = root.get(0);
        assertTrue(first.has("name"));
        assertTrue(first.has("params"));
    }

    private Path resolveOutPath() throws URISyntaxException {
        Path loc = Paths.get(
                SourcingCriteriaSettingGenerator.class
                        .getProtectionDomain()
                        .getCodeSource()
                        .getLocation()
                        .toURI()
        );
        Path baseDir = Files.isRegularFile(loc) ? loc.getParent() : loc;
        return baseDir.resolve("SourcingCriteriaSetting.json");
    }
}
