/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.context.DefaultSourcingContext;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.context.model.Product;
import com.fluentcommerce.util.sourcing.profile.SourcingStrategy.SourcingStrategyCondition;
import com.fluentretail.api.model.attribute.Attribute;
import com.google.common.collect.ImmutableList;
import lombok.Builder;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import java.time.Instant;
import java.util.*;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.*;

class SourcingConditionUtilsTest {

    private static final String OPERATOR_EXISTS = "exists";
    private static final String OPERATOR_EQUALS = "equals";
    private static final String OPERATOR_IN = "in";
    private static final String OPERATOR_NOT_IN = "not_in";
    private static final String OPERATOR_BETWEEN = "between";
    private static final String PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED = "fulfilmentChoice.pickupLocation.attributes.byName.certificationRequired";
    private static final String PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF = "unfulfilledItems.product.categories.ref";
    private static final String PATH_TOTALPRICE = "totalPrice";
    private static final String SCOPE_ANY = "ANY";
    private static final String VALUE_100 = "100";
    private static final String VALUE_UNFULFILLEDCATEGORY1 = "unfulfilledCategory1";


    @ParameterizedTest
    @MethodSource("provideValidConditionResponseList")
    void should_return_true_when_all_conditions_are_satisfied(List<SourcingStrategyCondition> conditionResponseList, boolean expectedResult) {

        // given
        SourcingContext sourcingContext = getSourcingContext();
        // when
        boolean result = SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, conditionResponseList);
        // then
        assertEquals(expectedResult, result);
    }


    @ParameterizedTest
    @MethodSource("provideInvalidConditionResponseList")
    void should_return_false_when_any_condition_fails(List<SourcingStrategyCondition> conditionResponseList, boolean expectedResult) {

        // given
        SourcingContext sourcingContext = getSourcingContext();
        // when
        boolean result = SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, conditionResponseList);
        // then
        assertEquals(expectedResult, result);
    }

    @ParameterizedTest
    @MethodSource("provideNullOperandsConditionResponseList")
    void should_return_false_when_any_operands_is_null(List<SourcingStrategyCondition> conditionResponseList, boolean expectedResult) {

        // given
        SourcingContext sourcingContext = getSourcingContext();
        // when
        boolean result = SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, conditionResponseList);
        // then
        assertEquals(expectedResult, result);
    }

    @Test
    void should_return_true_when_no_conditions_provided() {

        // given
        SourcingContext sourcingContext = getSourcingContext();
        // when
        boolean result = SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, new ArrayList<>());
        // then
        assertTrue(result);
    }

    @Test
    void should_throw_exception_when_sourcing_context_is_null() {

        // given
        SourcingStrategyCondition conditionResponse = SourcingStrategyCondition.builder()
                .type("fc.sourcing.condition.unregistered")
                .params(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_IN, VALUE_100))
                .build();
        // when
        NullPointerException ex = assertThrows(NullPointerException.class, () -> {
            SourcingConditionUtils.evaluateSourcingConditions(null, Arrays.asList(conditionResponse));
        });

        // then
        assertTrue(ex.getMessage().contains("sourcing context cannot be null"));
    }

    @Test
    void should_throw_exception_when_sourcing_condition_type_is_not_registered() {

        // given
        SourcingContext sourcingContext = getSourcingContext();
        SourcingStrategyCondition conditionResponse = SourcingStrategyCondition.builder()
                .type("fc.sourcing.condition.unregistered")
                .params(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_IN, VALUE_100))
                .build();
        // when
        IllegalArgumentException ex = assertThrows(IllegalArgumentException.class, () -> {
            SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, Arrays.asList(conditionResponse));
        });

        // then
        assertTrue(ex.getMessage().contains("Unknown SourcingCondition type: fc.sourcing.condition.unregistered"));

    }

    @Test
    void should_throw_exception_when_condition_type_is_unsupported() {
        // given
        SourcingContext sourcingContext = getSourcingContext();
        SourcingStrategyCondition conditionResponse = getConditionResponse(
                getConditionResponseParams(PATH_TOTALPRICE, "EQ", VALUE_100));

        // when
        IllegalArgumentException ex = assertThrows(IllegalArgumentException.class, () -> {
            SourcingConditionUtils.evaluateSourcingConditions(sourcingContext, Arrays.asList(conditionResponse));
        });

        // then
        assertTrue(ex.getMessage().contains("Unknown operator name: EQ"));
    }


    private SourcingContext getSourcingContext() {

        // prepare unfulfilled categories
        Product.Category unfulfilledCategory1 = Product.Category.builder().ref(VALUE_UNFULFILLEDCATEGORY1).build();
        Product.Category unfulfilledCategory2 = Product.Category.builder().ref("unfulfilledCategory2").build();
        Product.Category unfulfilledCategory3 = Product.Category.builder().ref("unfulfilledCategory3").build();
        Product.Category unfulfilledCategory4 = Product.Category.builder().ref("unfulfilledCategory4").build();


        OrderItem unfulfilledItem1 = OrderItem.builder().product(Product.builder()
                .categories(Arrays.asList(unfulfilledCategory1, unfulfilledCategory2)).build()).build();
        OrderItem unfulfilledItem2 = OrderItem.builder().product(Product.builder()
                .categories(Arrays.asList(unfulfilledCategory3, unfulfilledCategory4)).build()).build();

        // prepare fulfilmentChoice.pickupLocation.attributes.byName.certificationRequired
        Attribute attribute1 = Attribute.builder().name("certificationRequired").value(true).build();
        Location pickupLocation = Location.builder().attributes(Collections.singletonList(attribute1)).build();

        SourcingContext sourcingContext = DefaultSourcingContext.builder()
                .totalPrice(100d)
                .createdOn(Date.from(Instant.parse("2025-07-01T00:00:00.000Z")))
                .unfulfilledItems(Arrays.asList(unfulfilledItem1, unfulfilledItem2))
                .fulfilmentChoice(FulfilmentChoice.builder().pickupLocation(pickupLocation).build()).build();

        return sourcingContext;
    }


    private static Stream<Arguments> provideValidConditionResponseList() {

        return Stream.of(
                Arguments.of(
                        ImmutableList.of(
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_EQUALS, VALUE_100)),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "not_equals", "101")),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_IN, Collections.singletonList(VALUE_100))),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_NOT_IN, Collections.singletonList("101"))),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "greater_than", "99")),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "greater_than_or_equals", VALUE_100)),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "less_than", "101")),
                                getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "less_than_or_equals", VALUE_100))
                        ),
                        true
                ),
                Arguments.of(
                        ImmutableList.of(
                                getConditionResponse(getConditionResponseParams(
                                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_IN, Collections.singletonList("unfulfilledCategory3"), SCOPE_ANY)),
                                getConditionResponse(getConditionResponseParams(
                                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_EQUALS, "unfulfilledCategory3", SCOPE_ANY)),
                                getConditionResponse(getConditionResponseParams(
                                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, "in",
                                        Arrays.asList(VALUE_UNFULFILLEDCATEGORY1, "unfulfilledCategory2"), SCOPE_ANY)),
                                getConditionResponse(getConditionResponseParams(
                                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_NOT_IN,
                                        Arrays.asList("missedCategory1", "missedCategory2"), "ALL"))
                        ),
                        true
                ),
                Arguments.of(
                        ImmutableList.of(
                                getConditionResponse(getBetweenConditionResponseParams(PATH_TOTALPRICE, OPERATOR_BETWEEN, 90, 110))
                        ),
                        true
                ),
                Arguments.of(
                        ImmutableList.of(
                                getConditionResponse(getConditionResponseParams(
                                        PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_EQUALS, true)),
                                getConditionResponse(getConditionResponseParams(
                                        PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_IN, Collections.singletonList(true))),
                                getConditionResponse(getConditionResponseParams(
                                        PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_EXISTS, true)),
                                getConditionResponse(getConditionResponseParams(
                                        "fulfilmentChoice.pickupLocation.attributes.byName.MISSEDAttribute", OPERATOR_EXISTS, false))
                        ),
                        true
                ),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(
                        "createdOn", OPERATOR_BETWEEN, "2025-04-01T00:00:00.000Z", "2025-10-01T00:00:00.000Z"))), true)
        );
    }

    private static Stream<Arguments> provideInvalidConditionResponseList() {
        return Stream.of(

                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_EQUALS, "90"))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_IN, Collections.singletonList("90")))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_NOT_IN, Collections.singletonList(VALUE_100)))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "greater_than", VALUE_100))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "greater_than_or_equals", "110"))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "less_than", VALUE_100))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, "less_than_or_equals", "90"))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_EQUALS,
                        "missedCategory", SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_IN,
                        Collections.singletonList("missedCategory"), SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, "in",
                        Arrays.asList("MISSED1", "MISSED2"), SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, OPERATOR_NOT_IN,
                        Arrays.asList(VALUE_UNFULFILLEDCATEGORY1, VALUE_UNFULFILLEDCATEGORY1), "ALL"))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(PATH_TOTALPRICE, OPERATOR_BETWEEN, 90, 99))), false),

                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams(PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_IN, Collections.singletonList(false)))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams(PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_EQUALS, false))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams("fulfilmentChoice.pickupLocation.attributes.byName.MISSEDAttribute", OPERATOR_EXISTS, true))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(
                        "createdOn", OPERATOR_BETWEEN, "2025-04-01T00:00:00.000Z", "2025-05-01T00:00:00.000Z"))), false)
        );
    }

    private static Stream<Arguments> provideNullOperandsConditionResponseList() {
        return Stream.of(

                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams("missed", OPERATOR_EQUALS, VALUE_100))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams("missed", OPERATOR_IN, Collections.singletonList(VALUE_100)))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(PATH_TOTALPRICE, OPERATOR_NOT_IN, null))), false),

                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        "unfulfilledItems.product.categories.MISSED", OPERATOR_IN,
                        Collections.singletonList(VALUE_UNFULFILLEDCATEGORY1), SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        "unfulfilledItems.product.categories.MISSED", OPERATOR_EQUALS,
                        VALUE_UNFULFILLEDCATEGORY1, SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        PATH_UNFULFILLEDITEMS_PRODUCT_CATEGORIES_REF, "in",
                        null, SCOPE_ANY))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getConditionResponseParams(
                        "unfulfilledItems.product.categories.MISSED", OPERATOR_NOT_IN,
                        Arrays.asList("missedCategory1", null), "ALL"))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(PATH_TOTALPRICE, OPERATOR_BETWEEN, null, 110))), true),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(PATH_TOTALPRICE, OPERATOR_BETWEEN, null, null))), false),

                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams(PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_IN, null))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams(PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_EQUALS, null))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(
                        getConditionResponseParams(PATH_FILMENTCHOICE_PICKUPLOCATION_ATTRIBUTES_BYNAME_CERTIFICATIONREQUIRED, OPERATOR_EXISTS, null))), false),
                Arguments.of(ImmutableList.of(getConditionResponse(getBetweenConditionResponseParams(
                        "createdOn", OPERATOR_BETWEEN, "2025-04-01T00:00:00.000Z", null))), true)
        );
    }

    @Builder(builderClassName = "Builder", toBuilder = true)
    private static class ConditionResponseParams {
        String path;
        String operator;
        Object value;
        String conditionScope;
    }

    @Builder(builderClassName = "Builder", toBuilder = true)
    private static class ConditionResponseFromToValue {
        Object from;
        Object to;
    }

    private static SourcingStrategyCondition getConditionResponse(ObjectNode params) {
        return SourcingStrategyCondition.builder()
                .type(SourcingConditionTypeRegistry.FC_SOURCING_CONDITION_PATH_TYPE)
                .params(params)
                .build();
    }

    private static ObjectNode getConditionResponseParams(String path, String operator, Object value) {
        return getConditionResponseParams(path, operator, value, null);
    }

    private static ObjectNode getConditionResponseParams(String path, String operator, Object value, String conditionScope) {

        ConditionResponseParams params = ConditionResponseParams.builder()
                .path(path)
                .operator(operator)
                .value(value)
                .conditionScope(conditionScope).build();
        return JsonUtils.objectToNode(params);
    }

    private static ObjectNode getBetweenConditionResponseParams(String path, String operator, Object from, Object to) {

        ConditionResponseFromToValue fromToValue = ConditionResponseFromToValue.builder()
                .from(from)
                .to(to)
                .build();

        ConditionResponseParams params = ConditionResponseParams.builder()
                .path(path)
                .operator(operator)
                .value(JsonUtils.objectToNode(fromToValue))
                .build();
        return JsonUtils.objectToNode(params);
    }
}


