/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;

import com.fluentcommerce.graphql.sourcing.queries.inventory.GetVirtualPositionsQuery;
import com.fluentcommerce.util.sourcing.context.DefaultSourcingContext;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.Fulfilment.FulfilmentItem;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.context.model.Product;
import com.fluentcommerce.util.sourcing.criterion.InventoryAvailabilityCriterion;
import com.fluentcommerce.util.sourcing.criterion.BaseSourcingCriterion;
import com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils;
import com.fluentcommerce.util.sourcing.criterion.SourcingCriterion;
import com.fluentcommerce.util.test.executor.MockApiClient;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Answers;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.fluentcommerce.util.sourcing.model.Constants.ACTIVE;
import static com.fluentcommerce.util.sourcing.OrderUtils.itemsMinusFulfilments;
import static com.fluentcommerce.util.sourcing.SourcingUtils.Fulfilment;
import static com.fluentcommerce.util.sourcing.SourcingUtils.LocationAndPositions;
import static com.fluentcommerce.util.sourcing.SourcingUtils.findHighestValuePartialFulfilment;
import static com.fluentcommerce.util.sourcing.SourcingUtils.getLocationRefs;
import static com.fluentcommerce.util.sourcing.SourcingUtils.searchPermutationsForAllMatchPlan;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.Mockito.when;

@ExtendWith(MockitoExtension.class)
class SourcingUtilsTest {

    public static final String LOC_3 = "loc-3";
    private static final String AH8050_002 = "AH8050-002";

    private static final String PRODUCT1 = "PRODUCT1";
    private static final String PRODUCT2 = "PRODUCT2";
    private static final String LOCATION1 = "LOCATION1";
    private static final String LOCATION2 = "LOCATION2";


    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private Context context;

    @BeforeEach
    public void resetCache() {
        LocationUtils.resetCacheForTest();
    }

    @Test
    void testLocationAndPositionsComparable() {
        LocationAndPositions a = new LocationAndPositions(
                Location.builder()
                        .ref("a")
                        .name("1")
                        .primaryAddress(Address.builder()
                                .latitude(0d)
                                .longitude(0d)
                                .build())
                        .build(),
                null,
                new float[]{0.5f, 0.5f});
        LocationAndPositions b = new LocationAndPositions(
                Location.builder()
                        .ref("b")
                        .name("2")
                        .primaryAddress(Address.builder()
                                .latitude(0d)
                                .longitude(0d)
                                .build())
                        .build(),
                null,
                new float[]{0.5f, 0.7f});
        LocationAndPositions c = new LocationAndPositions(
                Location.builder()
                        .ref("c")
                        .name("3")
                        .primaryAddress(Address.builder()
                                .latitude(0d)
                                .longitude(0d)
                                .build())
                        .build(),
                null,
                new float[]{0.3f, 0.7f});

        assertEquals(ImmutableList.of(c, a, b),
                ImmutableList.of(a, b, c).stream().sorted().collect(Collectors.toList())
        );
    }


    private LocationAndPositions testLoc(String ref, int... values) {
        return new LocationAndPositions(
                Location.builder()
                        .ref(ref)
                        .name(ref)
                        .primaryAddress(Address.builder()
                                .latitude(0d)
                                .longitude(0d)
                                .build())
                        .build(),
                values,
                new float[]{0.5f});
    }

    @Test
    void testBreadthFirstSearchAlgorithm_singleStore() {
        ImmutableList<LocationAndPositions> plan = searchPermutationsForAllMatchPlan(
                ImmutableList.of(
                        testLoc("1", 0, 1),
                        testLoc("2", 6, 1),
                        testLoc("3", 6, 2),
                        testLoc("4" /* no stock == test will error if it doesn't efficiently return early */),
                        testLoc("5", 3, 3)
                ), 4,
                6, 2);

        assertEquals(1, plan.size());
        assertEquals("3", plan.get(0).getLocation().getRef());
    }

    @Test
    void testBreadthFirstSearchAlgorithm_threeStoreCombo() {
        ImmutableList<LocationAndPositions> plan = searchPermutationsForAllMatchPlan(
                ImmutableList.of(
                        testLoc("1", 0, 1),
                        testLoc("2", 6, 1),
                        testLoc("3", 6, 2),
                        testLoc("4", 6, 2),
                        testLoc("5", 3, 3)
                ), 4,
                12, 5);

        assertEquals(3, plan.size());
        assertEquals("1", plan.get(0).getLocation().getRef());
        assertEquals("3", plan.get(1).getLocation().getRef());
        assertEquals("4", plan.get(2).getLocation().getRef());
    }

    @Test
    void testItemsMinusFulfilments() {
        List<OrderItem> items = ImmutableList.of(
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("a")
                                .build())
                        .quantity(3)
                        .build(),
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("b")
                                .build())
                        .quantity(3)
                        .build(),
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("c")
                                .build())
                        .quantity(3)
                        .build()
        );

        List<Fulfilment> fulfilments = ImmutableList.of(
                new Fulfilment(null, ImmutableList.of(
                        FulfilmentItem.builder()
                                .orderItem(OrderItem.builder()
                                        .product(Product.builder()
                                                .ref("a")
                                                .build())
                                        .quantity(1)
                                        .build())
                                .requestedQuantity(1)
                                .build(),
                        FulfilmentItem.builder()
                                .orderItem(OrderItem.builder()
                                        .product(Product.builder()
                                                .ref("b")
                                                .build())
                                        .quantity(1)
                                        .build())
                                .requestedQuantity(1)
                                .build()
                ), "HD_PFS"),
                new Fulfilment(null, ImmutableList.of(
                        FulfilmentItem.builder()
                                .orderItem(OrderItem.builder()
                                        .product(Product.builder()
                                                .ref("a")
                                                .build())
                                        .quantity(2)
                                        .build())
                                .requestedQuantity(2)
                                .build(),
                        FulfilmentItem.builder()
                                .orderItem(OrderItem.builder()
                                        .product(Product.builder()
                                                .ref("b")
                                                .build())
                                        .quantity(1)
                                        .build())
                                .requestedQuantity(1)
                                .build()
                ), "HD_PFS")
        );

        List<OrderItem> result = itemsMinusFulfilments(items, fulfilments);

        assertEquals(3, result.size());
        assertEquals(0, result.get(0).getQuantity());
        assertEquals(1, result.get(1).getQuantity());
        assertEquals(3, result.get(2).getQuantity());
    }

    private List<LocationAndPositions> prepareLocationAndPositionsList(List<LocationAndPositions> laps,
                                                                       SourcingContext sourcingContext,
                                                                       SourcingCriteriaUtils.CriteriaArray sourcingCriteria) {
        List<LocationAndPositions> preparedLaps = laps.stream()
                .map(lap -> new LocationAndPositions(
                        lap.getLocation(),
                        lap.getQuantities(),
                        sourcingCriteria.apply(
                                SourcingCriteriaUtils.DefaultCriterionContext.builder()
                                        .sourcingContext(sourcingContext)
                                        .lap(lap)
                                        .build()
                        )
                ))
                .collect(toList());
        sourcingCriteria.normalize(preparedLaps.stream()
                .map(LocationAndPositions::getRating)
                .collect(toList()));
        return preparedLaps.stream()
                .sorted()
                .collect(collectingAndThen(toList(), ImmutableList::copyOf));
    }

    @Test
    void testSingleHighestValuePartialFulfilment() {
        ImmutableList<OrderItem> items = ImmutableList.of(
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("prod-a")
                                .build())
                        .quantity(5)
                        .paidPrice(1.0)
                        .build(),
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("prod-b")
                                .build())
                        .quantity(5)
                        .paidPrice(1.0)
                        .build()
        );

        final BaseSourcingCriterion availabilityCriterion = new InventoryAvailabilityCriterion();
        final SourcingCriteriaUtils.CriteriaArray criteriaArray = SourcingCriteriaUtils.CriteriaArray.of(availabilityCriterion);
        List<LocationAndPositions> laps = prepareLocationAndPositionsList(
                ImmutableList.of(
                        testLoc("loc-1", 3, 3),
                        testLoc("loc-2", 1, 0),
                        testLoc(LOC_3, 4, 3)
                ),
                DefaultSourcingContext.builder()
                        .unfulfilledItems(items)
                        .build(),
                criteriaArray
        );

        Optional<Fulfilment> fulfilment = findHighestValuePartialFulfilment(
                laps,
                items,
                ImmutableSet.of()
        );

        assertTrue(fulfilment.isPresent());
        assertEquals(LOC_3, fulfilment.get().getLocation().getRef());
        assertEquals(4, fulfilment.get().getItems().get(0).getRequestedQuantity());
        assertEquals(3, fulfilment.get().getItems().get(1).getRequestedQuantity());
    }

    @Test
    void testMultiHighestValuePartialFulfilment() {
        ImmutableList<OrderItem> items = ImmutableList.of(
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("prod-a")
                                .build())
                        .quantity(3)
                        .paidPrice(1.0)
                        .build(),
                OrderItem.builder()
                        .product(Product.builder()
                                .ref("prod-b")
                                .build())
                        .quantity(5)
                        .paidPrice(1.0)
                        .build()
        );

        final BaseSourcingCriterion availabilityCriterion = new InventoryAvailabilityCriterion();
        final SourcingCriteriaUtils.CriteriaArray criteriaArray = SourcingCriteriaUtils.CriteriaArray.of(availabilityCriterion);
        List<LocationAndPositions> laps = prepareLocationAndPositionsList(
                ImmutableList.of(
                        testLoc("loc-1", 3, 3),
                        testLoc("loc-2", 1, 0),
                        testLoc(LOC_3, 4, 3)
                ),
                DefaultSourcingContext.builder()
                        .unfulfilledItems(items)
                        .build(),
                criteriaArray
        );

        List<Fulfilment> fulfilments = new ArrayList<>();
        for (int i = 0; i < 3; i++) {
            List<OrderItem> remainingItems = itemsMinusFulfilments(items, fulfilments);
            if (remainingItems.stream().mapToInt(OrderItem::getQuantity).sum() <= 0) break;

            findHighestValuePartialFulfilment(
                    laps,
                    remainingItems,
                    getLocationRefs(fulfilments) // without this we'll get two fulfilments for loc-1
            ).ifPresent(fulfilments::add);
        }

        assertEquals(2, fulfilments.size());
        assertEquals(LOC_3, fulfilments.get(0).getLocation().getRef());
        assertEquals(3, fulfilments.get(0).getItems().get(0).getRequestedQuantity());
        assertEquals(3, fulfilments.get(0).getItems().get(1).getRequestedQuantity());
        assertEquals("loc-1", fulfilments.get(1).getLocation().getRef());
        assertEquals(2, fulfilments.get(1).getItems().get(0).getRequestedQuantity());
    }

    @Test
    void should_fetch_positions_unchanged_when_loading_positions_without_inventory_processor() {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/GetVirtualPositions.json");
        when(context.api()).thenReturn(api.get());

        final List<LocationAndPositions> laps = SourcingUtils.loadPositions(
                context,
                DefaultSourcingContext.builder()
                        .unfulfilledItems(Collections.singletonList(
                                OrderItem.builder()
                                        .product(Product.builder()
                                                .ref(AH8050_002)
                                                .build())
                                        .quantity(3)
                                        .paidPrice(1.0)
                                        .build()))
                        .build(),
                SourcingCriteriaUtils.CriteriaArray.of(),
                "catalogue",
                Collections.singletonList(Location.builder()
                        .ref("F_NSYD")
                        .build()),
                Collections.singletonList(ACTIVE),
                null
        );
        assertEquals(1, laps.size());
        assertEquals(1, laps.get(0).getQuantities().length);
        assertEquals(100, laps.get(0).getQuantities()[0]);
    }

    @Test
    void should_fetch_positions_changed_when_loading_positions_with_inventory_processor() {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/GetVirtualPositions.json");
        when(context.api()).thenReturn(api.get());

        final List<LocationAndPositions> laps = SourcingUtils.loadPositions(
                context,
                DefaultSourcingContext.builder()
                        .unfulfilledItems(Collections.singletonList(
                                OrderItem.builder()
                                        .product(Product.builder()
                                                .ref(AH8050_002)
                                                .build())
                                        .quantity(3)
                                        .paidPrice(1.0)
                                        .build()))
                        .build(),
                SourcingCriteriaUtils.CriteriaArray.of(),
                "catalogue",
                Collections.singletonList(Location.builder()
                        .ref("F_NSYD")
                        .build()),
                Collections.singletonList(ACTIVE),
                (sourcingContext, positions) -> positions.stream()
                        .map(position -> {
                            if (AH8050_002.equals(position.getProductRef())) {
                                return position.toBuilder()
                                        .quantity(50)
                                        .build();
                            } else {
                                return position;
                            }
                        })
                        .collect(Collectors.toList())
        );
        assertEquals(1, laps.size());
        assertEquals(1, laps.get(0).getQuantities().length);
        assertEquals(50, laps.get(0).getQuantities()[0]);
    }

    @ParameterizedTest
    @MethodSource("provideLocationOrderTestData")
    void should_return_correct_location_order_when_different_products_are_processed(final String vpositionsFileName,
                                                                                    final SourcingContext sourcingContext,
                                                                                    final SourcingCriteriaUtils.CriteriaArray criteriaArray,
                                                                                    final List<Location> locationList,
                                                                                    final List<LocationAndPositions> locationAndPositionsResultList) {
        // given
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/" + vpositionsFileName);
        when(context.api()).thenReturn(api.get());

        // when
        final List<LocationAndPositions> locationAndPositionsList = SourcingUtils.loadPositions(
                context,
                sourcingContext,
                criteriaArray,
                "catalogue",
                locationList,
                Collections.singletonList(ACTIVE),
                null
        );
        // then
        assertEqualLocationAndPositionsLists(locationAndPositionsResultList, locationAndPositionsList);
    }

    private static Stream<Arguments> provideLocationOrderTestData() {
        final String VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION = "GetVirtualPositionsOneProductOneCatalogOneLocation.json";

        //we do not test a case where one product in two catalogues has the same location.
        return Stream.of(

                //1)PRODUCT1 with an unfulfilled quantity of 25 exists in one catalogs, with 100 units , located at one location.
                Arguments.of(
                        VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION,
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 25)),
                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1), toArray(1f)), getAtsDependentCriterion()),
                        getLocationList(LOCATION1),
                        Arrays.asList(getLocationAndPositions(LOCATION1, toArray(100), toArray(1f, 1f)))),

                //1.1) only ATS agnostic criterion
                Arguments.of(
                        VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION,
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 25)),
                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1), toArray(1f))),
                        getLocationList(LOCATION1),
                        Arrays.asList(getLocationAndPositions(LOCATION1, toArray(100), toArray(1f)))),

                //1.2) only ATS dependent criterion
                Arguments.of(
                        VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION,
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 25)),
                        getCriteriaArray(getAtsDependentCriterion()),
                        getLocationList(LOCATION1),
                        Arrays.asList(getLocationAndPositions(LOCATION1, toArray(100), toArray(1f)))),
                //1.3) without criteria
                Arguments.of(
                        VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION,
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 25)),
                        getCriteriaArray(),
                        getLocationList(LOCATION1),
                        Arrays.asList(getLocationAndPositions(LOCATION1, toArray(100), new float[]{}))),


                //2) PRODUCT1 with an unfulfilled quantity of 200 exists as two positions with 100 units in each in one catalog, located at two different locations.
                Arguments.of(
                        "GetVirtualPositionsOneProductOneCatalogTwoLocations.json",
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 200)),

                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1, LOCATION2), toArray(0f, 1f)), getAtsDependentCriterion()),
                        getLocationList(LOCATION1, LOCATION2),
                        Arrays.asList(
                                getLocationAndPositions(LOCATION2, toArray(100), toArray(1f, 1f)),
                                getLocationAndPositions(LOCATION1, toArray(100), toArray(0f, 1f)))),
                //3) PRODUCT1 and PRODUCT2 with an unfulfilled quantity of 200 and 50 exists as two positions with 100 units in each in one catalog, located at one location.
                Arguments.of(
                        "GetVirtualPositionsTwoProductsOneCatalogOneLocation.json",
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 200), getOrderItem(PRODUCT2, 50)),

                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1), toArray(0.1f)), getAtsDependentCriterion()),
                        getLocationList(LOCATION1),
                        Arrays.asList(
                                getLocationAndPositions(LOCATION1, toArray(100, 100), toArray(1f, 1f)))),

                //4) PRODUCT1 and PRODUCT2 with an unfulfilled quantity of 200 and 50 exists as two positions with 100 units in each in one catalog, located at two locations.
                Arguments.of(
                        "GetVirtualPositionsTwoProductsOneCatalogTwoLocations.json",
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 200), getOrderItem(PRODUCT2, 50)),

                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1, LOCATION2), toArray(0.1f, 0.2f)), getAtsDependentCriterion()),
                        getLocationList(LOCATION1, LOCATION2),
                        Arrays.asList(
                                getLocationAndPositions(LOCATION2, toArray(0, 100), toArray(1f, 1f)),
                                getLocationAndPositions(LOCATION1, toArray(100, 0), toArray(0f, 1f)))),

                //5)PRODUCT1 with ratings that excludes location
                Arguments.of(
                        VPOSITIONS_ONE_PRODUCT_ONE_CATALOG_ONE_LOCATION,
                        getSourcingContextWithUnfulfilledItems(getOrderItem(PRODUCT1, 25)),
                        getCriteriaArray(getAtsAgnosticCriterion(getLocationList(LOCATION1), toArray(SourcingCriterion.RATING_EXCLUDE))),
                        getLocationList(LOCATION1),
                        Collections.emptyList())

        );
    }

    private static SourcingCriteriaUtils.CriteriaArray getCriteriaArray(final SourcingCriterion... criteria) {
        return SourcingCriteriaUtils.CriteriaArray.of(criteria);
    }

    private static List<Location> getLocationList(final String... locationNames) {

        List<Location> locations = new ArrayList<>();

        for (String locationName : locationNames) {
            locations.add(Location.builder().ref(locationName).build());
        }
        return locations;
    }

    private static int[] toArray(final int... integers) {
        return integers;
    }

    private static float[] toArray(final float... integers) {
        return integers;
    }

    private static LocationAndPositions getLocationAndPositions(final String locationRef, final int[] quantities, final float[] rating) {
        Location location = Location.builder().ref(locationRef).build();
        return LocationAndPositions.builder().location(location).quantities(quantities).rating(rating).build();
    }

    private static void assertEqualLocationAndPositionsLists(List<LocationAndPositions> expectedList,
                                                             List<LocationAndPositions> actualList) {
        assertThat(actualList).hasSameSizeAs(expectedList);

        for (int i = 0; i < expectedList.size(); i++) {
            LocationAndPositions expected = expectedList.get(i);
            LocationAndPositions actual = actualList.get(i);

            assertThat(actual.getLocation().getRef())
                    .isEqualTo(expected.getLocation().getRef());

            assertThat(actual.getQuantities())
                    .as("Quantities at index %d", i)
                    .containsExactly(expected.getQuantities());

            assertThat(actual.getRating())
                    .as("Ratings at index %d", i)
                    .usingComparatorWithPrecision(0.001f)
                    .containsExactly(expected.getRating());
        }
    }

    private static SourcingCriterion getAtsAgnosticCriterion(final List<Location> locations, final float[] ratings) {
        if (locations.size() != ratings.length) {
            throw new IllegalArgumentException("locations and ratings must have the same size");
        }

        Map<String, Float> locationToRatingMap = new HashMap<>();
        for (int i = 0; i < locations.size(); i++) {
            locationToRatingMap.put(locations.get(i).getRef(), ratings[i]);
        }

        return context -> {
            String locationRef = context.getLap().getLocation().getRef();
            return locationToRatingMap.getOrDefault(locationRef, 0f);
        };
    }

    private static SourcingCriterion getAtsDependentCriterion() {
        return new InventoryAvailabilityCriterion();
    }

    private static OrderItem getOrderItem(final String productRef, final int quantity) {
        OrderItem orderItem = OrderItem.builder()
                .product(Product.builder()
                        .ref(productRef)
                        .build())
                .quantity(quantity)
                .build();

        return orderItem;
    }

    private static SourcingContext getSourcingContextWithUnfulfilledItems(final OrderItem... orderItems) {

        return DefaultSourcingContext.builder()
                .unfulfilledItems(Arrays.asList(orderItems))
                .build();
    }
}
