/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;

import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentretail.rubix.v2.context.Context;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.Collections;
import java.util.List;

import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.CENTIMETRES;
import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.KILOMETRES;
import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.METRES;
import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.MILES;
import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;


class LocationUtilsTest {

    @Mock
    private Context mockContext;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
        LocationUtils.resetCacheForTest();
    }

    @AfterEach
    void tearDown() {
        LocationUtils.resetCacheForTest();
    }

    @Test
    void should_return_correct_distances_when_all_units_are_used() {
        // Test coordinates: from (0,0) to (1,1) - approximately 157 km
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 1;
        final double lon2 = 1;
        
        // Test all distance units
        final double km = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, KILOMETRES);
        final double miles = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, MILES);
        final double metres = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, METRES);
        
        // Verify that results are positive and reasonable
        assertTrue(km > 0, "Kilometres should be positive");
        assertTrue(miles > 0, "Miles should be positive");
        assertTrue(metres > 0, "Metres should be positive");
        
        // Check relationships between units
        assertEquals(metres, km * 1000, 0.001, "Metres should equal kilometres * 1000");
        
        // Check approximate km/miles ratio (1 mile ≈ 1.609 km)
        final double expectedKmFromMiles = miles * 1.609;
        assertEquals(km, expectedKmFromMiles, 1.0, "Kilometres should approximately equal miles * 1.609");
    }

    @Test
    void should_return_zero_when_calculating_distance_between_same_points() {
        final double lat = 0;
        final double lon = 0;
        
        // Distance from point to itself should be 0
        assertEquals(0.0, LocationUtils.calculateDistance(lat, lon, lat, lon, KILOMETRES), 0.001);
        assertEquals(0.0, LocationUtils.calculateDistance(lat, lon, lat, lon, MILES), 0.001);
        assertEquals(0.0, LocationUtils.calculateDistance(lat, lon, lat, lon, METRES), 0.001);
    }

    @Test
    void should_throw_exception_when_unsupported_unit_is_provided() {
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 1;
        final double lon2 = 1;
        
        // CENTIMETRES is not supported in switch, should throw exception
        assertThrows(IllegalArgumentException.class, () -> {
            LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, CENTIMETRES);
        }, "Unsupported unit should throw IllegalArgumentException");
    }

    @Test
    void should_return_same_results_when_using_calculateDistance_and_individual_methods() {
        // Test consistency with individual methods
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 0.5;
        final double lon2 = 0.5;

        final double kmFromCalculate = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, KILOMETRES);
        final double kmFromDirect = LocationUtils.distanceInKilometres(lat1, lon1, lat2, lon2);
        assertEquals(kmFromDirect, kmFromCalculate, 0.001, "calculateDistance should match distanceInKilometres");

        final double milesFromCalculate = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, MILES);
        final double milesFromDirect = LocationUtils.distanceInMiles(lat1, lon1, lat2, lon2);
        assertEquals(milesFromDirect, milesFromCalculate, 0.001, "calculateDistance should match distanceInMiles");

        final double metresFromCalculate = LocationUtils.calculateDistance(lat1, lon1, lat2, lon2, METRES);
        final double metresFromDirect = LocationUtils.distanceInMetres(lat1, lon1, lat2, lon2);
        assertEquals(metresFromDirect, metresFromCalculate, 0.001, "calculateDistance should match distanceInMetres");
    }

    @Test
    void should_return_correct_distance_in_metres() {
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 1;
        final double lon2 = 1;
        
        final double result = LocationUtils.distanceInMetres(lat1, lon1, lat2, lon2);
        final double expectedKm = LocationUtils.distanceInKilometres(lat1, lon1, lat2, lon2);
        
        assertTrue(result > 0, "Distance in metres should be positive");
        assertEquals(expectedKm * 1000, result, 0.001, "Metres should equal kilometres * 1000");
    }

    @Test
    void should_return_correct_distance_in_kilometres() {
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 1;
        final double lon2 = 1;
        
        final double result = LocationUtils.distanceInKilometres(lat1, lon1, lat2, lon2);
        
        assertTrue(result > 0, "Distance in kilometres should be positive");
        assertTrue(result < 200, "Distance should be reasonable (less than 200 km for 1 degree)");
    }

    @Test
    void should_return_correct_distance_in_miles() {
        final double lat1 = 0;
        final double lon1 = 0;
        final double lat2 = 1;
        final double lon2 = 1;
        
        final double result = LocationUtils.distanceInMiles(lat1, lon1, lat2, lon2);
        final double expectedKm = LocationUtils.distanceInKilometres(lat1, lon1, lat2, lon2);
        
        assertTrue(result > 0, "Distance in miles should be positive");
        assertTrue(result < 150, "Distance should be reasonable (less than 150 miles for 1 degree)");
        
        // Check approximate conversion (1 mile ≈ 1.609 km)
        final double expectedMilesFromKm = expectedKm / 1.609;
        assertEquals(expectedMilesFromKm, result, 1.0, "Miles should approximately equal kilometres / 1.609");
    }

    @Test
    void should_return_zero_distance_when_same_coordinates_are_provided() {
        final double lat = 0;
        final double lon = 0;
        
        assertEquals(0.0, LocationUtils.distanceInMetres(lat, lon, lat, lon), 0.001);
        assertEquals(0.0, LocationUtils.distanceInKilometres(lat, lon, lat, lon), 0.001);
        assertEquals(0.0, LocationUtils.distanceInMiles(lat, lon, lat, lon), 0.001);
    }

    @Test
    void should_reset_cache_successfully() {
        // This test verifies that resetCacheForTest() doesn't throw exceptions
        assertDoesNotThrow(() -> LocationUtils.resetCacheForTest());
    }

    @Test
    void should_return_null_when_getLocationByRef_with_null_reference() {
        final String nullRef = null;
        
        final Location result = LocationUtils.getLocationByRef(mockContext, nullRef);
        
        assertNull(result, "Should return null when location reference is null");
    }

    @Test
    void should_return_empty_list_when_getLocationsInNetworks_with_empty_list() {
        final List<String> emptyNetworkRefs = Collections.emptyList();
        
        final List<Location> result = LocationUtils.getLocationsInNetworks(mockContext, emptyNetworkRefs);
        
        assertNotNull(result, "Result should not be null");
        assertTrue(result.isEmpty(), "Result should be empty for empty network references");
    }
}
