/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.SourcingUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParam;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParamSelectComponentOption;
import com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits;

import java.util.Optional;

import static com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils.getFloats;
import static com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils.getDistanceUnits;
import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.KILOMETRES;
import static com.fluentcommerce.util.sourcing.units.DistanceMeasurementUnits.MILES;


/**
 * Example of the generated JSON based on annotations:
 * {
 *     "name": "locationDistanceBanded",
 *     "type": "fc.sourcing.criterion.locationDistanceBanded",
 *     "tags": [
 *       "ATS-agnostic"
 *     ],
 *     "params": [
 *       {
 *         "name": "value",
 *         "component": "fc.field.multistring",
 *         "mandatory": "true",
 *         "type": "number"
 *       }
 *     ]
 *   }
 */

@SourcingCriterionInfo(name = "locationDistanceBanded",
        type = "fc.sourcing.criterion.locationDistanceBanded",
        tags = {"ATS-agnostic"})
public class LocationDistanceBandedCriterion extends BaseSourcingCriterion {

    @SourcingCriterionParam(name = "value", component = "fc.field.multistring", type = "number")
    private Float[] bands;

    @SourcingCriterionParam(
            name = "valueUnit",
            component = "select",
            selectComponentOptions = {
                    @SourcingCriterionParamSelectComponentOption(
                            label = "fc.sourcing.criterion.locationDistanceBanded.valueunit.kilometres",
                            value = "kilometres"
                    ),
                    @SourcingCriterionParamSelectComponentOption(
                            label = "fc.sourcing.criterion.locationDistanceBanded.valueunit.miles",
                            value = "miles"
                    )
            }
    )
    private DistanceMeasurementUnits distanceUnit;

    @Override
    public void parseParams(final JsonNode params) {

        bands = getFloats(params, "value");

        distanceUnit = getDistanceUnits(params, "valueUnit");
    }

    @Override
    protected float execute(final SourcingCriteriaUtils.CriterionContext criterionContext) {

        if (distanceUnit != KILOMETRES && distanceUnit != MILES) {
            return 0;
        }

        final Double lat = Optional.ofNullable(criterionContext.getSourcingContext())
                .map(SourcingContext::getFulfilmentChoice)
                .map(FulfilmentChoice::getAddress)
                .map(Address::getLatitude)
                .orElse(null);
        final Double lon = Optional.ofNullable(criterionContext.getSourcingContext())
                .map(SourcingContext::getFulfilmentChoice)
                .map(FulfilmentChoice::getAddress)
                .map(Address::getLongitude)
                .orElse(null);

        if (lat == null || lon == null) {
            return 0;
        }

        if (bands.length == 0) {
            return 0;
        }

        final Double locationLat = Optional.ofNullable(criterionContext.getLap())
                .map(SourcingUtils.LocationAndPositions::getLocation)
                .map(Location::getPrimaryAddress)
                .map(Address::getLatitude)
                .orElse(null);
        final Double locationLon = Optional.ofNullable(criterionContext.getLap())
                .map(SourcingUtils.LocationAndPositions::getLocation)
                .map(Location::getPrimaryAddress)
                .map(Address::getLongitude)
                .orElse(null);

        if (locationLat == null || locationLon == null) {
            return 0;
        }

        final double distance = LocationUtils.calculateDistance(lat, lon, locationLat, locationLon, distanceUnit);

        for (int i = bands.length - 1; i >= 0; i--) {
            if (distance > bands[i]) return (bands.length - 1) - i;
        }
        return bands.length;
    }
}
