/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParam;

import java.util.List;
import java.util.Optional;

@SourcingCriterionInfo(name = "inventoryAvailabilityExclusion",
        type = "fc.sourcing.criterion.inventoryAvailabilityExclusion",
        tags = {"Exclusion", "ATS-dependent"})
public class InventoryAvailabilityExclusionCriterion extends BaseSourcingCriterion {

    @SourcingCriterionParam(name = "value", component = "integer")
    private Double threshold;

    @Override
    public void parseParams(final JsonNode params) {
        threshold = Optional.ofNullable(params)
                .map(p -> p.get("value"))
                .filter(JsonNode::isNumber)
                .map(JsonNode::asDouble)
                .orElse(null);
    }

    @Override
    protected float execute(final SourcingCriteriaUtils.CriterionContext criterionContext) {
        if (criterionContext.getLap().getQuantities() == null) {
            return 0;
        }

        if (threshold == null) {
            return 0;
        }

        final List<OrderItem> items = criterionContext.getSourcingContext().getUnfulfilledItems();

        int total = 0;
        int filled = 0;

        for (int i = 0; i < items.size(); i++) {
            int requested = items.get(i).getQuantity();
            total += requested;
            filled += criterionContext.getLap().getQuantities()[i];
        }
        if (total == 0) {
            return 1f;
        }
        return (((float) filled) / total) * 100.0 >= threshold ? 1f : -1f;
    }

}
