/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParam;

import java.util.List;

import static com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils.getFloats;

/**
 * Example of the generated JSON based on annotations:
 * {
 *     "name": "inventoryAvailabilityBanded",
 *     "type": "fc.sourcing.criterion.inventoryAvailabilityBanded",
 *     "tags": [
 *       "ATS-dependent"
 *     ],
 *     "params": [
 *       {
 *         "name": "value",
 *         "component": "fc.field.multistring",
 *         "mandatory": "true",
 *         "type": "number"
 *       }
 *     ]
 *   }
 */

@SourcingCriterionInfo(name = "inventoryAvailabilityBanded",
        type = "fc.sourcing.criterion.inventoryAvailabilityBanded",
        tags = {"ATS-dependent"})
public class InventoryAvailabilityBandedCriterion extends BaseSourcingCriterion {

    @SourcingCriterionParam(name = "value", component = "fc.field.multistring", type = "number")
    private Float[] bands;

    @Override
    public void parseParams(final JsonNode params) {
        bands = getFloats(params, "value");
    }

    @Override
    protected float execute(final SourcingCriteriaUtils.CriterionContext criterionContext) {
        if (criterionContext.getLap().getQuantities() == null) {
            return 0;
        }

        if (bands.length == 0) {
            return 0;
        }

        final List<OrderItem> items = criterionContext.getSourcingContext().getUnfulfilledItems();

        int total = 0;
        int filled = 0;

        for (int i = 0; i < items.size(); i++) {
            int requested = items.get(i).getQuantity();
            total += requested;
            filled += criterionContext.getLap().getQuantities()[i];
        }
        if (total == 0) {
            return 1f;
        }
        final float availability = (((float) filled) / total) * 100.0f;

        for (int i = bands.length - 1; i >= 0; i--) {
            if (availability > bands[i]) return i + 1;
        }
        return 0;
    }

}
