/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParam;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParamSelectComponentOption;
import lombok.NoArgsConstructor;

import java.lang.reflect.Field;

/**
 * Basic implementation of the {@link SourcingCriterion}.
 *
 * <p>Extensions of this abstract class define specific business rules that are
 * evaluated dynamically during the sourcing process. The actual logic of the criterion is implemented in
 * the 'execute' method. This implementation also offers ways to specify pre- and post-execution logic.</p>
 */
@NoArgsConstructor
public abstract class BaseSourcingCriterion implements SourcingCriterion {

    /**
     * Applies the criterion function to a criterion context by calling the override of the 'execute' method.
     *
     * @param criterionContext context of criterion execution, including Sourcing Context and location information
     * @return "rating" of the location in the context
     */
    public float apply(final SourcingCriteriaUtils.CriterionContext criterionContext) {
        preExecution(criterionContext);
        final float result = execute(criterionContext);
        postExecution(criterionContext, result);
        return result;
    }

    protected void preExecution(final SourcingCriteriaUtils.CriterionContext criterionContext) {
    }

    protected void postExecution(final SourcingCriteriaUtils.CriterionContext criterionContext, final float result) {
    }

    protected abstract float execute(final SourcingCriteriaUtils.CriterionContext criterionContext);

    /**
     * Builds a JSON representation of the sourcing criterion, including its metadata and parameter definitions,
     * based on annotations present on the class and its fields.
     * <p>
     * The generated JSON includes the following structure:
     * <ul>
     *   <li><b>name</b> - the internal name of the criterion</li>
     *   <li><b>type</b> - the unique identifier of the criterion type</li>
     *   <li><b>tags</b> - any associated tags for filtering or categorization</li>
     *   <li><b>label</b> - optional label for UI display</li>
     *   <li><b>description</b> - optional description text</li>
     *   <li><b>params</b> - a list of parameter definitions extracted from annotated fields</li>
     * </ul>
     * <p>
     * Each parameter may also include an optional <b>extensions</b> field for additional metadata.
     *
     * @return an {@link ObjectNode} representing the JSON structure of the criterion
     */
    public ObjectNode toJson() {
        ObjectMapper mapper = new ObjectMapper();
        ObjectNode root = mapper.createObjectNode();

        SourcingCriterionInfo annotation = this.getClass().getAnnotation(SourcingCriterionInfo.class);
        if (annotation == null) {
            return root;
        }

        root.put("name", annotation.name());
        root.put("type", annotation.type());

        addTags(annotation, root, mapper);
        addOptionalFields(annotation, root);
        addParams(root, mapper);

        return root;
    }

    private void addTags(SourcingCriterionInfo annotation, ObjectNode root, ObjectMapper mapper) {
        ArrayNode tagsArray = mapper.createArrayNode();
        for (String tag : annotation.tags()) {
            tagsArray.add(tag);
        }
        root.set("tags", tagsArray);
    }

    private void addOptionalFields(SourcingCriterionInfo annotation, ObjectNode root) {
        if (!annotation.label().isEmpty()) {
            root.put("label", annotation.label());
        }
        if (!annotation.description().isEmpty()) {
            root.put("description", annotation.description());
        }
    }

    private void addParams(ObjectNode root, ObjectMapper mapper) {
        ArrayNode paramsArray = mapper.createArrayNode();

        for (Field field : getClass().getDeclaredFields()) {
            if (field.isAnnotationPresent(SourcingCriterionParam.class)) {
                SourcingCriterionParam param = field.getAnnotation(SourcingCriterionParam.class);
                ObjectNode paramNode = mapper.createObjectNode();

                paramNode.put("name", param.name());
                paramNode.put("component", param.component());
                paramNode.put("mandatory", param.mandatory());

                if (param.exactSearch()) {
                    paramNode.put("exactSearch", true);
                }

                if (param.type() != null && !param.type().isEmpty()) {
                    paramNode.put("type", param.type());
                }

                if (param.selectComponentOptions().length > 0) {
                    paramNode.set("options", buildSelectOptions(param.selectComponentOptions(), mapper));
                }

                paramsArray.add(paramNode);
            }
        }

        root.set("params", paramsArray);
    }

    private ArrayNode buildSelectOptions(SourcingCriterionParamSelectComponentOption[] selectOptions, ObjectMapper mapper) {
        ArrayNode optionsArray = mapper.createArrayNode();
        for (SourcingCriterionParamSelectComponentOption option : selectOptions) {
            ObjectNode optionNode = mapper.createObjectNode();
            optionNode.put("label", option.label());
            optionNode.put("value", option.value());
            optionsArray.add(optionNode);
        }
        return optionsArray;
    }
}
