/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context.loader;

import com.fluentcommerce.graphql.sourcing.queries.order.GetOrderForSourcingQuery;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.DefaultSourcingContext;
import com.fluentcommerce.util.sourcing.context.UnfulfilledItemProcessor;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.Customer;
import com.fluentcommerce.util.sourcing.context.model.Fulfilment;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.context.model.Product;
import com.fluentretail.api.model.sku.Price;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.sourcing.model.Constants.ENTITY_SUBTYPE_ORDER_HD;
import static com.fluentcommerce.util.sourcing.model.Constants.ENTITY_TYPE_FULFILMENT_CHOICE;
import static com.fluentcommerce.util.sourcing.model.Constants.ENTITY_TYPE_ORDER;
import static com.fluentcommerce.util.sourcing.context.SourcingContextUtils.mapAttributes;

/**
 * {@link SourcingContextLoader} used for loading context for "ORDER" entity type
 */
public class OrderContextLoader extends AbstractSourcingContextLoader<GetOrderForSourcingQuery.Data> {
    @Override
    protected GetOrderForSourcingQuery.Data loadResponse(final Context context) {
        // CC or HD Order workflows - Orders have a single Fulfilment Choice
        return (GetOrderForSourcingQuery.Data)
                context.api().query(GetOrderForSourcingQuery.builder()
                        .id(context.getEvent().getEntityId())
                        .build());
    }

    @Override
    protected SourcingContext mapResponse(final GetOrderForSourcingQuery.Data orderResponse,
                                          final Context context,
                                          final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        final SourcingContext sourcingContext = map(orderResponse.order(), context);
        final Location pickupLocation = LocationUtils.getLocationByRef(
                context, Optional.ofNullable(sourcingContext)
                        .map(SourcingContext::getFulfilmentChoice)
                        .map(FulfilmentChoice::getPickupLocationRef)
                        .orElse(null));

        return Optional.ofNullable(sourcingContext)
                .map(SourcingContext::toBuilder)
                .map(builder -> builder
                        .fulfilmentChoice(Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                .map(FulfilmentChoice::toBuilder)
                                .map(fulfilmentChoiceBuilder -> fulfilmentChoiceBuilder
                                        .pickupLocation(pickupLocation)
                                        .address(Objects.equals(sourcingContext.getType(), ENTITY_SUBTYPE_ORDER_HD) ?
                                                Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                                        .map(FulfilmentChoice::getDeliveryAddress)
                                                        .orElse(null) :
                                                Optional.ofNullable(pickupLocation)
                                                        .map(Location::getPrimaryAddress)
                                                        .orElseGet(() -> Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                                                .map(FulfilmentChoice::getDeliveryAddress)
                                                                .orElse(null))
                                        )
                                        .build())
                                .orElse(null))
                        .unfulfilledItems(unfulfilledItemProcessor.process(sourcingContext)).build())
                .orElse(null);
    }

    protected SourcingContext map(final GetOrderForSourcingQuery.Order order, final Context context) {
        if (order == null) {
            return DefaultSourcingContext.builder().build();
        }
        final FulfilmentChoice fulfilmentChoice = mapFulfilmentChoice(order.fulfilmentChoices(), context);
        final List<Fulfilment> fulfilments = mapFulfilments(order.fulfilments());
        // collect order items that either belong to no Fulfilment Choice or belong to the current one
        final List<OrderItem> orderItems = Optional.ofNullable(mapItems(order.items()))
                .map(items -> items.stream()
                        .filter(item -> item.getFulfilmentChoice() == null ||
                                Objects.equals(item.getFulfilmentChoice().getRef(),
                                        Optional.ofNullable(fulfilmentChoice)
                                                .map(FulfilmentChoice::getRef)
                                                .orElse(null)))
                        .collect(Collectors.toList()))
                .orElse(null);

        return DefaultSourcingContext.builder()
                .id(order.id())
                .ref(order.ref())
                .ref2(order.ref2())
                .tag1(order.tag1())
                .tag2(order.tag2())
                .tag3(order.tag3())
                .type(order.type())
                .status(order.status())
                .createdOn((Date) order.createdOn())
                .updatedOn((Date) order.updatedOn())
                .totalPrice(order.totalPrice())
                .totalTaxPrice(order.totalTaxPrice())
                .attributes(mapAttributes(order.attributes()))
                .items(orderItems)
                .fulfilments(fulfilments)
                .customer(JsonUtils.anyToPojo(order.customer(), Customer.class))
                .fulfilmentChoice(fulfilmentChoice)
                .build();
    }

    private List<OrderItem> mapItems(final GetOrderForSourcingQuery.Items items) {
        if (items == null || items.itemEdges() == null || items.itemEdges().isEmpty()) {
            return null;
        }
        return items.itemEdges().stream()
                .map(GetOrderForSourcingQuery.ItemEdge::itemNode)
                .map(itemNode -> OrderItem.builder()
                        .id(itemNode.id())
                        .ref(itemNode.ref())
                        .quantity(itemNode.quantity())
                        .currency(itemNode.currency())
                        .price(itemNode.price())
                        .paidPrice(itemNode.paidPrice())
                        .totalPrice(itemNode.totalPrice())
                        .totalTaxPrice(itemNode.totalTaxPrice())
                        .taxPrice(itemNode.taxPrice())
                        .status(itemNode.status())
                        .product(mapProduct(itemNode.product()))
                        .attributes(mapAttributes(itemNode.attributes()))
                        .fulfilmentChoice(JsonUtils.anyToPojo(itemNode.fulfilmentChoice(), FulfilmentChoice.class))
                        .build())
                .collect(Collectors.toList());
    }

    private Product mapProduct(final GetOrderForSourcingQuery.Product product) {
        if (product == null) {
            return null;
        }
        return Product.builder()
                .id(product.id())
                .createdOn((Date) product.createdOn())
                .updatedOn((Date) product.updatedOn())
                .ref(product.ref())
                .type(product.type())
                .status(product.status())
                .name(product.name())
                .summary(product.summary())
                .attributes(mapAttributes(product.attributes()))
                .prices(Optional.ofNullable(product.prices())
                        .map(prices -> prices.stream()
                                .map(price -> JsonUtils.anyToPojo(price, Price.class))
                                .collect(Collectors.toList()))
                        .orElse(null)
                )
                .tax(Optional.ofNullable(product.tax())
                        .map(tax -> Product.Tax.builder()
                                .country(tax.country())
                                .group(tax.group())
                                .tariff(tax.tariff())
                                .build())
                        .orElse(null))
                .categories(mapCategories(product.categories()))
                .catalogueRef(product.catalogue().ref())
                .build();
    }

    private List<Product.Category> mapCategories(GetOrderForSourcingQuery.Categories categories) {
        if (categories == null || categories.categoryEdges() == null || categories.categoryEdges().isEmpty()) {
            return null;
        }
        return categories.categoryEdges().stream()
                .map(GetOrderForSourcingQuery.CategoryEdge::categoryNode)
                .map(categoryNode -> JsonUtils.anyToPojo(categoryNode, Product.Category.class))
                .collect(Collectors.toList());
    }

    private List<Fulfilment> mapFulfilments(final GetOrderForSourcingQuery.Fulfilments fulfilments) {
        return Optional.ofNullable(fulfilments)
                .map(GetOrderForSourcingQuery.Fulfilments::fulfilmentEdges)
                .map(edges -> edges.stream()
                        .map(edge -> mapFulfilmentNode(edge.fulfilmentNode()))
                        .collect(Collectors.toList()))
                .orElse(null);
    }

    private Fulfilment mapFulfilmentNode(final GetOrderForSourcingQuery.FulfilmentNode node) {
        if (node == null) {
            return null;
        }
        final Map<String, List<Fulfilment.FulfilmentItem.RejectedItem>> rejectedItemMap = mapRejectedItems(node.rejections());
        return Fulfilment.builder()
                .id(node.id())
                .ref(node.ref())
                .status(node.status())
                .fulfilmentChoiceRef(node.fulfilmentChoiceRef())
                .createdOn((Date) node.createdOn())
                .updatedOn((Date) node.updatedOn())
                .deliveryType(node.deliveryType())
                .type(node.type())
                .eta(node.eta())
                .expiryTime((Date) node.expiryTime())
                .fromLocation(JsonUtils.anyToPojo(node.fromLocation(), Fulfilment.LocationLink.class))
                .fromAddress(JsonUtils.anyToPojo(node.fromAddress(), Address.class))
                .toAddress(JsonUtils.anyToPojo(node.toAddress(), Address.class))
                .user(JsonUtils.anyToPojo(node.user(), Customer.class))
                .attributes(mapAttributes(node.attributes()))
                .items(Optional.ofNullable(mapFulfilmentItems(node.fulfilmentItems()))
                        .map(fulfilmentItems -> fulfilmentItems.stream()
                                .map(fulfilmentItem -> fulfilmentItem.toBuilder()
                                        .rejectedItems(rejectedItemMap.get(fulfilmentItem.getId()))
                                        .build())
                                .collect(Collectors.toList()))
                        .orElse(null)
                )
                .build();
    }

    private List<Fulfilment.FulfilmentItem> mapFulfilmentItems(final GetOrderForSourcingQuery.FulfilmentItems fulfilmentItems) {
        return Optional.ofNullable(fulfilmentItems)
                .map(GetOrderForSourcingQuery.FulfilmentItems::fulfilmentItemEdges)
                .map(edges -> edges.stream()
                        .map(edge -> mapFulfilmentItemNode(edge.fulfilmentItemNode()))
                        .collect(Collectors.toList()))
                .orElse(null);
    }

    private Fulfilment.FulfilmentItem mapFulfilmentItemNode(final GetOrderForSourcingQuery.FulfilmentItemNode itemNode) {
        return Fulfilment.FulfilmentItem.builder()
                .id(itemNode.id())
                .ref(itemNode.ref())
                .status(itemNode.status())
                .requestedQuantity(itemNode.requestedQuantity())
                .filledQuantity(itemNode.filledQuantity())
                .rejectedQuantity(itemNode.rejectedQuantity())
                .orderItem(JsonUtils.anyToPojo(itemNode.orderItem(), OrderItem.class))
                .attributes(mapAttributes(itemNode.attributes()))
                .build();
    }

    private Map<String, List<Fulfilment.FulfilmentItem.RejectedItem>> mapRejectedItems(final GetOrderForSourcingQuery.Rejections rejections) {
        final Map<String, List<Fulfilment.FulfilmentItem.RejectedItem>> rejectedItems = new HashMap<>();
        if (rejections != null && rejections.rejectionEdges() != null) {
            for (GetOrderForSourcingQuery.RejectionEdge edge : rejections.rejectionEdges()) {
                final GetOrderForSourcingQuery.RejectionNode node = edge.rejectionNode();
                if (node != null) {
                    rejectedItems.computeIfAbsent(node.fulfilmentItem().id(), k -> new ArrayList<>())
                            .add(JsonUtils.anyToPojo(node, Fulfilment.FulfilmentItem.RejectedItem.class));
                }
            }
        }
        return rejectedItems;
    }

    private FulfilmentChoice mapFulfilmentChoice(final GetOrderForSourcingQuery.FulfilmentChoices fulfilmentChoices,
                                                 final Context context) {
        if (fulfilmentChoices == null || fulfilmentChoices.fulfilmentChoicesEdges() == null
                || fulfilmentChoices.fulfilmentChoicesEdges().isEmpty()) {
            return null;
        }
        GetOrderForSourcingQuery.FulfilmentChoicesNode node = null;
        final Event event = context.getEvent();
        if (ENTITY_TYPE_ORDER.equalsIgnoreCase(event.getEntityType())) {
            // CC or HD Order has a single Fulfilment Choice
            node = Optional.ofNullable(fulfilmentChoices.fulfilmentChoicesEdges())
                    .map(fulfilmentChoicesEdges -> fulfilmentChoicesEdges.get(0))
                    .map(GetOrderForSourcingQuery.FulfilmentChoicesEdge::fulfilmentChoicesNode)
                    .orElse(null);
        } else if (ENTITY_TYPE_FULFILMENT_CHOICE.equalsIgnoreCase(event.getEntityType())) {
            // Multi Order has multiple Fulfilment Choices, need to get the one being processed at the moment
            node = fulfilmentChoices.fulfilmentChoicesEdges().stream()
                    .map(GetOrderForSourcingQuery.FulfilmentChoicesEdge::fulfilmentChoicesNode)
                    .filter(fulfilmentChoicesNode -> Objects.equals(fulfilmentChoicesNode.ref(), event.getEntityRef()))
                    .findFirst()
                    .orElse(null);
        }
        if (node == null) {
            return null;
        }
        return FulfilmentChoice.builder()
                .id(node.id())
                .ref(node.ref())
                .type(node.type())
                .status(node.status())
                .createdOn((Date) node.createdOn())
                .updatedOn((Date) node.updatedOn())
                .currency(node.currency())
                .deliveryFirstName(node.deliveryFirstName())
                .deliveryLastName(node.deliveryLastName())
                .deliveryContact(node.deliveryContact())
                .deliveryEmail(node.deliveryEmail())
                .deliverAfter((Date) node.deliverAfter())
                .deliverBefore((Date) node.deliverBefore())
                .dispatchOn((Date) node.dispatchOn())
                .deliveryInstruction(node.deliveryInstruction())
                .deliveryType(node.deliveryType())
                .fulfilmentTaxPrice(node.fulfilmentTaxPrice())
                .fulfilmentPrice(node.fulfilmentPrice())
                .fulfilmentType(node.fulfilmentType())
                .pickupLocationRef(node.pickupLocationRef())
                .deliveryAddress(JsonUtils.anyToPojo(node.deliveryAddress(), Address.class))
                .attributes(mapAttributes(node.attributes()))
                .build();
    }
}
