/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context.loader;

import com.fluentcommerce.graphql.sourcing.queries.order.GetOrderForSourcingQuery;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.UnfulfilledItemProcessor;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;

import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.sourcing.model.Constants.ENTITY_SUBTYPE_ORDER_HD;

/**
 * {@link SourcingContextLoader} used for loading context for "FULFILMENT_CHOICE" entity type.
 * Loads the parent Order object and only takes into account the relevant Fulfilment Choice
 */
public class FulfilmentChoiceLoader extends OrderContextLoader {

    @Override
    protected GetOrderForSourcingQuery.Data loadResponse(final Context context) {
        // Multi Order workflow - Orders have multiple Fulfilment Choices, but only one is being processed at the moment
        final Event event = context.getEvent();
        return (GetOrderForSourcingQuery.Data)
                context.api().query(GetOrderForSourcingQuery.builder()
                        .id(event.getRootEntityId())
                        .fulfilmentChoiceRef(Collections.singletonList(event.getEntityRef()))
                        .build());
    }

    @Override
    protected SourcingContext mapResponse(final GetOrderForSourcingQuery.Data orderResponse,
                                          final Context context,
                                          final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        final SourcingContext sourcingContext = map(orderResponse.order(), context);
        final Location pickupLocation = LocationUtils.getLocationByRef(
                context, Optional.ofNullable(sourcingContext)
                        .map(SourcingContext::getFulfilmentChoice)
                        .map(FulfilmentChoice::getPickupLocationRef)
                        .orElse(null));

        return Optional.ofNullable(sourcingContext)
                .map(SourcingContext::toBuilder)
                .map(builder -> builder
                        .totalPrice(sourcingContext.getItems() == null
                                ? sourcingContext.getTotalPrice()
                                : (Double) sourcingContext.getItems().stream()
                                .filter(orderItem -> Objects.nonNull(orderItem.getTotalPrice()))
                                .mapToDouble(OrderItem::getTotalPrice)
                                .sum())
                        .totalTaxPrice(sourcingContext.getItems() == null
                                ? sourcingContext.getTotalTaxPrice()
                                : (Double) sourcingContext.getItems().stream()
                                .filter(orderItem -> Objects.nonNull(orderItem.getTotalTaxPrice()))
                                .mapToDouble(OrderItem::getTotalTaxPrice)
                                .sum())
                        .fulfilmentChoice(Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                .map(FulfilmentChoice::toBuilder)
                                .map(fulfilmentChoiceBuilder -> fulfilmentChoiceBuilder
                                        .pickupLocation(pickupLocation)
                                        .address(Objects.equals(sourcingContext.getFulfilmentChoice().getType(), ENTITY_SUBTYPE_ORDER_HD) ?
                                                Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                                        .map(FulfilmentChoice::getDeliveryAddress)
                                                        .orElse(null) :
                                                Optional.ofNullable(pickupLocation)
                                                        .map(Location::getPrimaryAddress)
                                                        .orElseGet(() -> Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                                                .map(FulfilmentChoice::getDeliveryAddress)
                                                                .orElse(null))
                                        )
                                        .build())
                                .orElse(null))
                        // leave only fulfilments related to the processed Fulfilment Choice
                        .fulfilments(Optional.ofNullable(sourcingContext.getFulfilments())
                                .map(fulfilments -> fulfilments.stream()
                                        .filter(fulfilment -> Objects.equals(
                                                fulfilment.getFulfilmentChoiceRef(),
                                                Optional.ofNullable(sourcingContext.getFulfilmentChoice())
                                                        .map(FulfilmentChoice::getRef)
                                                        .orElse(null)))
                                        .collect(Collectors.toList()))
                                .orElse(null))
                        .unfulfilledItems(unfulfilledItemProcessor.process(sourcingContext)).build())
                .orElse(null);
    }

}
