/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context.loader;

import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.UnfulfilledItemProcessor;
import com.fluentretail.rubix.v2.context.Context;

/**
 * Abstract base class for loading and mapping a {@link SourcingContext} from a given {@link Context}.
 * <p>
 * Subclasses are expected to provide specific implementations for how data is retrieved (via {@code loadResponse})
 * and how it is transformed into a {@link SourcingContext} (via {@code mapResponse}).
 * <p>
 * This abstraction is useful when sourcing logic needs to be adapted for different entity types
 * (e.g. Order, FulfilmentChoice), while maintaining a common pattern for building sourcing context.
 *
 * @param <T> the type of response data retrieved and used to build the {@link SourcingContext}
 */
public abstract class AbstractSourcingContextLoader<T> implements SourcingContextLoader {

    /**
     * Loads and constructs a {@link SourcingContext} using the provided {@link Context}
     * and an {@link UnfulfilledItemProcessor} to compute unfulfilled order items.
     *
     * @param context                  the rule execution context
     * @param unfulfilledItemProcessor helper for identifying unfulfilled items
     * @return a fully constructed {@link SourcingContext}
     */
    public SourcingContext load(final Context context,
                                final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        final T response = loadResponse(context);
        return mapResponse(response, context, unfulfilledItemProcessor);
    }

    protected abstract T loadResponse(Context context);

    protected abstract SourcingContext mapResponse(T response,
                                                   Context context,
                                                   UnfulfilledItemProcessor unfulfilledItemProcessor);

}
