/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;

import java.util.Map;

/**
 * Implementation of the {@link SourcingConditionOperator} that checks
 * if a context value lies between two bounds (inclusive).
 *
 * <p>This operator expects exactly two operands:
 * <ol>
 *   <li>The first operand is the context value to check; must be {@link Comparable} or {@code null}.</li>
 *   <li>The second operand is a {@link Map} with optional keys "from" and "to" representing
 *       the lower and upper bounds respectively. Either bound can be {@code null} to indicate
 *       an open-ended range.</li>
 * </ol>
 * The evaluation returns {@code true} if the context value is greater than or equal to
 * the "from" bound (if present) and less than or equal to the "to" bound (if present).</p>
 */
public class SourcingConditionBetweenOperator extends AbstractSourcingConditionOperator {

    @Override
    protected boolean doEvaluate(Object... operands) {

        Object contextValue = getContextValue(operands);

        if (contextValue == null || operands[1] == null) {
            return false;
        }

        Map<?, ?> value = (Map<?, ?>) operands[1];

        Object from = convertValue(value.get("from"), getContextValueType(operands));
        Object to = convertValue(value.get("to"), getContextValueType(operands));

        if (from == null && to == null) {
            return false;
        }

        boolean lowerCheck = from == null || toComparable(contextValue).compareTo(toComparable(from)) >= 0;
        boolean upperCheck = to == null || toComparable(contextValue).compareTo(toComparable(to)) <= 0;

        return lowerCheck && upperCheck;
    }


    @Override
    protected boolean validateOperands(Object... operands) {

        if (operands.length != BINARY_OPERATOR_OPERAND_COUNT) {
            return false;
        }

        if (!(operands[1] == null || operands[1] instanceof Map)) {
            return false;
        }

        Class<?> valueType = getContextValueType(operands);

        return valueType == null || Comparable.class.isAssignableFrom(valueType);
    }

    public static boolean ifValueBetween(Object... operands) {
        return new SourcingConditionBetweenOperator().evaluate(operands[0], operands[1]);
    }
}
