/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.profile.SourcingStrategy.SourcingStrategyCondition;
import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Utility class that provides helper methods for evaluating sourcing conditions.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing strategy execution.</p>
 */
public final class SourcingConditionUtils {

    private SourcingConditionUtils() {
    }

    /**
     * Evaluates a list of sourcing conditions against the given {@link SourcingContext}.
     *
     * <p>Each condition is converted to its corresponding {@link SourcingCondition} implementation,
     * deserialized from its parameters, and then evaluated against the sourcing context.</p>
     *
     * @param sourcingContext the context containing data relevant for sourcing condition evaluation; must not be {@code null}
     * @param conditions      the list of sourcing conditions to evaluate; may be {@code null} or empty
     * @return {@code true} if all conditions are satisfied or if the list is null/empty,
     * {@code false} if at least one condition fails
     * @throws NullPointerException if {@code sourcingContext} is {@code null}
     */
    public static boolean evaluateSourcingConditions(final SourcingContext sourcingContext,
                                                     final List<SourcingStrategyCondition> conditions) {

        Objects.requireNonNull(sourcingContext, "sourcing context cannot be null");

        if (CollectionUtils.isEmpty(conditions)) {
            return true;
        }

        List<SourcingCondition> sourcingConditionList = new ArrayList<>();

        JsonNode sourcingContextJson = JsonUtils.objectToNode(sourcingContext);

        for (SourcingStrategyCondition condition : conditions) {

            Class<? extends SourcingCondition> sourcingConditionType = SourcingConditionTypeRegistry.getSourcingConditionType(condition.getType());
            SourcingCondition sourcingCondition = JsonUtils.anyToPojo(condition.getParams(), sourcingConditionType);

            sourcingConditionList.add(sourcingCondition);
        }

        return sourcingConditionList.stream().allMatch(sourcingCondition -> sourcingCondition.evaluateWithContext(sourcingContextJson));
    }
}
