/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Registry for mapping sourcing condition type names to their corresponding implementation classes.
 *
 * <p>This utility class provides methods to register and retrieve sourcing condition types
 * by their string identifiers. It is used during sourcing strategy evaluation to dynamically
 * instantiate condition implementations based on type names.</p>
 *
 * <p>The registry is thread-safe and uses a {@link ConcurrentHashMap} to store mappings.</p>
 */
final class SourcingConditionTypeRegistry {

    private SourcingConditionTypeRegistry() {
    }

    private static final Map<String, Class<? extends SourcingCondition>> typeNameToClassMap = new ConcurrentHashMap<>();

    /**
     * Constant representing the default Fluent Commerce path-based sourcing condition type.
     */
    public static final String FC_SOURCING_CONDITION_PATH_TYPE = "fc.sourcing.condition.path";

    static {
        register(FC_SOURCING_CONDITION_PATH_TYPE, DefaultSourcingCondition.class);
    }

    /**
     * Registers a sourcing condition type with its corresponding implementation class.
     *
     * @param typeName the string identifier of the sourcing condition type
     * @param clazz    the class that implements {@link SourcingCondition} for the given type
     */
    public static void register(String typeName, Class<? extends SourcingCondition> clazz) {
        typeNameToClassMap.put(typeName, clazz);
    }

    /**
     * Retrieves the implementation class associated with the given sourcing condition type name.
     *
     * @param typeName the string identifier of the sourcing condition type
     * @return the class implementing {@link SourcingCondition} for the specified type name
     * @throws IllegalArgumentException if no class is registered for the given type name
     */
    public static Class<? extends SourcingCondition> getSourcingConditionType(String typeName) {

        Class<? extends SourcingCondition> clazz = typeNameToClassMap.get(typeName);
        if (clazz == null) {
            throw new IllegalArgumentException("Unknown SourcingCondition type: " + typeName);
        }

        return clazz;
    }
}
