/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.condition.operator.SourcingConditionOperator;
import com.fluentcommerce.util.sourcing.condition.operator.SourcingConditionOperatorRegistry;

import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static com.fluentcommerce.util.sourcing.condition.SourcingConditionConstants.SourcingConditionScope;

/**
 * Default implementation of the {@link SourcingCondition} interface.
 *
 * <p>This class evaluates a sourcing condition based on a JSON path, an operator, and a value.
 * It supports evaluation over single values as well as arrays of values within the sourcing context.</p>
 *
 * <p>If the JSON node at the specified path is an array, the condition is evaluated against each element
 * using the provided {@link SourcingConditionScope} (e.g. {@code ALL}, {@code ANY}, {@code NONE}).
 * Otherwise, the condition is evaluated directly against the single value.</p>
 *
 * <p>The supported operators include equality, comparison, set membership, and existence checks,
 * and are resolved via {@link SourcingConditionOperatorRegistry} using the operator string.</p>
 *
 * <p>This class is intended to be deserialized from JSON using Jackson, and can be registered in the
 * {@link SourcingConditionTypeRegistry} to handle conditions of a specific type.</p>
 *
 * @see SourcingCondition
 * @see SourcingConditionScope
 * @see SourcingConditionOperator
 * @see SourcingConditionOperatorRegistry
 */
public class DefaultSourcingCondition implements SourcingCondition {

    protected static final ObjectMapper mapper = new ObjectMapper();

    @JsonProperty
    protected String path;
    @JsonProperty
    protected String operator;
    @JsonProperty
    protected SourcingConditionScope conditionScope;
    @JsonProperty
    protected Object value;

    /**
     * Evaluates the sourcing condition against the provided JSON context.
     *
     * <p>If the value at the specified {@code path} is an array, the condition will be evaluated
     * for each element using the configured {@link SourcingConditionScope} (e.g. {@code ALL}, {@code ANY}, {@code NONE}).
     * Otherwise, the condition is evaluated directly against the single value.</p>
     *
     * <p>The evaluation uses the operator specified by the {@code operator} field and compares
     * the resolved value from the context to the expected {@code value}.</p>
     *
     * @param sourcingContext the {@link JsonNode} representing the input context (e.g. order, inventory, product data)
     * @return {@code true} if the condition is satisfied according to the scope and operator, {@code false} otherwise
     * @throws IllegalArgumentException if the condition scope or operator is missing or invalid,
     *                                  or if the path does not point to a value node
     */
    @Override
    public boolean evaluateWithContext(JsonNode sourcingContext) {

        JsonNode contextNode = JsonUtils.getPath(sourcingContext, path);

        if (contextNode.isArray()) {
            return doEvaluate(flatten(contextNode));
        } else {
            return doEvaluate(contextNode);
        }
    }

    protected boolean doEvaluate(Stream<JsonNode> contextNodeStream) {

        if (conditionScope == null) {
            throw new IllegalArgumentException("Condition scope can not be null");
        }

        switch (conditionScope) {

            case ALL:
                return contextNodeStream.allMatch(this::doEvaluate);
            case ANY:
                return contextNodeStream.anyMatch(this::doEvaluate);
            case NONE:
                return contextNodeStream.noneMatch(this::doEvaluate);
            default:
                throw new IllegalStateException("Unexpected condition scope: " + conditionScope);
        }
    }

    protected boolean doEvaluate(JsonNode contextNode) {

        return getSourcingConditionOperator().evaluate(getContextNodeValue(contextNode), value);
    }

    protected Object getContextNodeValue(JsonNode contextNode) {

        if (!contextNode.isValueNode()) {
            throw new IllegalArgumentException("Context node must be a value node");
        }

        return mapper.convertValue(contextNode, Object.class);

    }

    protected SourcingConditionOperator getSourcingConditionOperator() {

        return SourcingConditionOperatorRegistry.getSourcingConditionOperator(operator);
    }


    protected Stream<JsonNode> flatten(final JsonNode node) {
        if (node.isArray()) {
            return StreamSupport.stream(node.spliterator(), false)
                    .flatMap(this::flatten);
        } else {
            return Stream.of(node);
        }
    }
}
