/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;

import com.fluentcommerce.graphql.sourcing.queries.inventory.GetVirtualPositionsQuery;
import com.fluentcommerce.graphql.sourcing.queries.profile.GetProfileByRefQuery;
import com.fluentcommerce.graphql.type.VirtualCatalogueKey;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.condition.SourcingCondition;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Fulfilment.FulfilmentItem;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils;
import com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils.CriteriaArray;
import com.fluentcommerce.util.sourcing.inventory.InventoryProcessor;
import com.fluentcommerce.util.sourcing.inventory.VirtualPosition;
import com.fluentcommerce.util.sourcing.profile.Network;
import com.fluentcommerce.util.sourcing.profile.SourcingProfile;
import com.fluentcommerce.util.sourcing.profile.SourcingStrategy;
import com.fluentcommerce.util.sourcing.profile.VirtualCatalogue;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.Value;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.Stack;

import static com.fluentcommerce.util.sourcing.model.Constants.PROP_SOURCING_PROFILE_REF;
import static com.fluentcommerce.util.sourcing.model.Constants.ACTIVE;
import static com.fluentcommerce.util.sourcing.OrderUtils.itemsMinusFulfilments;
import static com.fluentcommerce.util.sourcing.condition.SourcingConditionUtils.evaluateSourcingConditions;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * Utility class that provides helper methods for working with Sourcing Profiles and executing sourcing logic.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing rule execution.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class SourcingUtils {

    private static final VirtualPosition NO_STOCK = VirtualPosition.builder()
            .ref("NO_STOCK")
            .status("INACTIVE")
            .quantity(0)
            .build();
    private static final int MAX_POSITIONS = 5000;

    public static final ImmutableList<LocationAndPositions> NONE = ImmutableList.of();

    /**
     * Loads a {@link SourcingProfile} based on the provided profile reference
     *
     * @param context rule context object
     * @return loaded Sourcing Profile
     */
    public static SourcingProfile loadSourcingProfile(final Context context) {
        final GetProfileByRefQuery.Data profileResponse = (GetProfileByRefQuery.Data)
                context.api().query(GetProfileByRefQuery.builder()
                        .profileRef(context.getProp(PROP_SOURCING_PROFILE_REF))
                        .build());

        return JsonUtils.anyToPojo(profileResponse.sourcingProfile(), SourcingProfile.class);
    }

    /**
     * Load Virtual Positions for locations that can fulfill the unfulfilled items with additional inventory processing
     *
     * @param context            rule context object
     * @param sourcingContext    {@link SourcingContext} object with unfulfilled items
     * @param sourcingCriteria   {@link CriteriaArray} of Sourcing Criteria to rank Locations
     * @param catalogueRef       Virtual Catalogue reference to fetch Positions from
     * @param locations          list of Locations for which Positions should be fetched
     * @param positionStatuses   statuses for which Virtual Positions should be fetched
     * @param inventoryProcessor {@link InventoryProcessor} that would process Virtual Positions after loading
     * @return a list of Virtual Positions ranked by Sourcing Criteria
     */
    public static List<LocationAndPositions> loadPositions(final Context context,
                                                           final SourcingContext sourcingContext,
                                                           final CriteriaArray sourcingCriteria,
                                                           final String catalogueRef,
                                                           final List<Location> locations,
                                                           final List<String> positionStatuses,
                                                           final InventoryProcessor inventoryProcessor
    ) {
        final List<OrderItem> items = sourcingContext.getUnfulfilledItems();
        if (items.isEmpty()) {
            return ImmutableList.of();
        }

        // first we sort the list of locations by the criteria, so we know which to prioritise when fetching stock
        // only ATS-agnostic criteria should work now
        List<LocationAndPositions> locationsWithoutStock = locations.stream()
                .map(location -> new LocationAndPositions(location,
                        null,
                        sourcingCriteria.apply(
                                SourcingCriteriaUtils.DefaultCriterionContext.builder()
                                        .sourcingContext(sourcingContext)
                                        .lap(new LocationAndPositions(location, null, null))
                                        .build()
                        )))
                .filter(LocationAndPositions::isValid)
                .collect(toList());
        // values of ratings should be normalized
        sourcingCriteria.normalize(locationsWithoutStock.stream()
                .map(LocationAndPositions::getRating)
                .collect(toList()));
        // sort after criteria ratings were normalized
        locationsWithoutStock = locationsWithoutStock.stream()
                .sorted(Comparator.reverseOrder())
                .collect(toList());

        // load positions for first [X] locations where [X] = MAX_POSITIONS / no. of items
        int locationLimit = MAX_POSITIONS / items.size();

        GetVirtualPositionsQuery.Data positions = (GetVirtualPositionsQuery.Data) context.api().query(
                GetVirtualPositionsQuery.builder()
                        .catalogues(Collections.singletonList(
                                VirtualCatalogueKey.builder()
                                        .ref(catalogueRef)
                                        .build()))
                        .products(items.stream().map(i -> i.getProduct().getRef()).distinct().collect(toList()))
                        .groups(locationsWithoutStock.stream()
                                .limit(locationLimit)
                                .map(lap -> lap.getLocation().getRef())
                                .collect(toList()))
                        .first(MAX_POSITIONS)
                        .status(positionStatuses)
                        .build());
        List<VirtualPosition> virtualPositions = JsonUtils.stream(positions.virtualPositions().edges())
                .map(GetVirtualPositionsQuery.Edge::node)
                .map(node -> JsonUtils.anyToPojo(node, VirtualPosition.class))
                .collect(toList());
        if (inventoryProcessor != null) {
            virtualPositions = inventoryProcessor.process(sourcingContext, virtualPositions);
        }

        // group into location fulfilments
        Map<String, Map<String, VirtualPosition>> collected = virtualPositions.stream()
                .collect(groupingBy(
                        VirtualPosition::getGroupRef,
                        mapping(node -> node, toMap(VirtualPosition::getProductRef, n -> n))
                ));

        // collect item quantities from loaded Virtual Positions
        // apply criteria again, both ats-agnostic and ats-dependent should work this time
        locationsWithoutStock = locationsWithoutStock.stream()
                .map(lap -> new LocationAndPositions(
                        lap.getLocation(),
                        items.stream().mapToInt(i ->
                                collected
                                        .getOrDefault(lap.getLocation().getRef(), ImmutableMap.of())
                                        .getOrDefault(i.getProduct().getRef(), NO_STOCK).getQuantity()
                        ).toArray(),
                        lap.getRating()
                ))
                .map(lap -> new LocationAndPositions(
                        lap.getLocation(),
                        lap.getQuantities(),
                        sourcingCriteria.apply(
                                SourcingCriteriaUtils.DefaultCriterionContext.builder()
                                        .sourcingContext(sourcingContext)
                                        .lap(lap)
                                        .build()
                        )
                ))
                .filter(LocationAndPositions::isValid)
                .collect(toList());
        // values of ratings should be normalized
        sourcingCriteria.normalize(locationsWithoutStock.stream()
                .map(LocationAndPositions::getRating)
                .collect(toList()));
        // sort after criteria ratings were normalized
        return locationsWithoutStock.stream()
                .sorted(Comparator.reverseOrder())
                .limit(locationLimit)
                .collect(collectingAndThen(toList(), ImmutableList::copyOf));
    }

    /**
     * Collects list of location references from the Fulfilments
     *
     * @param fulfilments list of fulfilments
     * @return list of location references from the Fulfilments
     */
    public static Set<String> getLocationRefs(List<Fulfilment> fulfilments) {
        return fulfilments.stream().map(f -> f.getLocation().getRef()).collect(toSet());
    }

    /**
     * Find the highest value partial fulfilment for the given items.
     * Each location in the provided list will be evaluated against the rating calculated by Sourcing Criteria, and the
     * fulfilment with the highest value will be returned.
     * This method can be looped multiple times to attempt to fulfil as much of an order as possible
     * when one or more rules using `findPlanForAllItems` have failed.
     *
     * @param locationsAndPositions list of locations with position information (see `loadPositions`)
     * @param remainingItems        list of items that need to be fulfilled
     * @param excludedLocations     Set of location refs to exclude from consideration.
     * @return the highest value proposed fulfilment, or none if there were no positive value fulfilments.
     */
    public static Optional<Fulfilment> findHighestValuePartialFulfilment(
            final List<LocationAndPositions> locationsAndPositions,
            final List<OrderItem> remainingItems,
            final Set<String> excludedLocations
    ) {

        final List<LocationAndPositions> filteredLaps = locationsAndPositions.stream()
                .filter(lap -> !excludedLocations.contains(lap.getLocation().getRef()))
                .collect(toList());

        final List<float[]> values = filteredLaps.stream()
                .map(LocationAndPositions::getRating)
                .collect(toList());

        float[] bestResult = new float[values.isEmpty() ? 1 : values.get(0).length];
        int bestIndex = -1;

        for (int i = 0; i < values.size(); i++) {
            if (arrayCompare(values.get(i), bestResult) > 0 &&
                    canFulfill(filteredLaps.get(i), remainingItems)) {
                bestIndex = i;
                bestResult = values.get(i);
            }
        }

        return (bestIndex >= 0)
                ? Optional.of(buildPlan(ImmutableList.of(filteredLaps.get(bestIndex)), remainingItems).getFulfilments().get(0))
                : Optional.empty();
    }

    private static boolean canFulfill(final LocationAndPositions lap,
                                      final List<OrderItem> remainingItems) {
        for (int i = 0; i < lap.getQuantities().length; i++) {
            if (lap.getQuantities()[i] > 0 && remainingItems.get(i).getQuantity() > 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * Find a set of fulfilments that completely fulfil the given items.
     * Locations will be considered in order from the highest to the lowest rating, as defined by the `sourcingCriteria`.
     * Plans split across fewer locations are _always_ considered better than those with more.
     * If no combination of eligible locations can fill the order, an empty list will be returned.
     *
     * @param context            the rule context object
     * @param catalogueRef       reference of the virtual catalogue containing position information
     * @param sourcingContext    Sourcing Context of the sourcing request
     * @param locations          list of locations to consider (see `LocationUtils`)
     * @param sourcingCriteria   array of criteria to evaluate the rating of locations. Earlier criteria
     *                           take precedence over later ones, and those that return a higher value are
     *                           considered better options than that return a lower one. Locations with a negative
     *                           rating are excluded from consideration.
     * @param maxSplit           max number of splits to perform - plans beyond this size won't be checked
     * @param positionStatuses   list of statuses for Virtual Positions to be considered for fulfilments
     * @param inventoryProcessor {@link InventoryProcessor} that would process Virtual Positions after loading
     * @return a `SourcingPlan` with the set of fulfilments that should be created (or no fulfilments if no
     * valid plan was found)
     */
    public static SourcingPlan findPlanForAllItems(final Context context,
                                                   final String catalogueRef,
                                                   final SourcingContext sourcingContext,
                                                   final List<Location> locations,
                                                   final CriteriaArray sourcingCriteria,
                                                   final int maxSplit,
                                                   final List<String> positionStatuses,
                                                   final InventoryProcessor inventoryProcessor
    ) {
        final List<OrderItem> items = sourcingContext.getUnfulfilledItems();
        if (items == null) {
            return buildPlan(new ArrayList<>(), new ArrayList<>());
        }
        if (items.isEmpty()) {
            return buildPlan(new ArrayList<>(), items);
        }

        final int[] quantities = items.stream()
                .mapToInt(OrderItem::getQuantity)
                .toArray();

        final List<LocationAndPositions> laps = loadPositions(
                context,
                sourcingContext,
                sourcingCriteria,
                catalogueRef,
                locations,
                positionStatuses,
                inventoryProcessor);

        // now we can try to find the best combo of laps to fill the order
        final List<LocationAndPositions> bestCombo =
                searchPermutationsForAllMatchPlan(laps, maxSplit, quantities);

        // finally, convert back into a Sourcing plan
        return buildPlan(bestCombo, items);
    }

    /**
     * A breadth-first search across locations for the best fit fulfilment plan.
     * The algorithm finds the first location or set of locations that can fulfil the items, giving priority to
     * solutions with better "rating" (according to the locations SourcingCriteria).
     * Fewer splits is considered always preferable to more splits.
     *
     * @param locations  List of locations to consider, including inventory info
     * @param maxSplit   maximum number of splits that can constitute one plan
     * @param quantities array of quantities to be fulfilled
     * @return the most efficient set of locations that can fulfil the items. Empty list means there is no viable plan.
     */
    public static ImmutableList<LocationAndPositions> searchPermutationsForAllMatchPlan(
            List<LocationAndPositions> locations,
            int maxSplit,
            int... quantities
    ) {
        // loop so that we check all singles, then pairs, triplets, etc
        final Stack<Integer> currentPlan = new Stack<>();
        final Stack<int[]> remainingQuantities = new Stack<>();
        for (int fulfilmentLimit = 1; fulfilmentLimit <= maxSplit + 1; fulfilmentLimit++) {
            // Queue for the fulfilment plan, we'll push and pop to try out all the permutations
            currentPlan.clear();

            // As we check permutations we'll "cache" our remainder calculations to avoid reprocessing
            remainingQuantities.clear();
            remainingQuantities.add(quantities); // start with the overall remaining items on the order/fo

            int loc = 0;
            while (loc < locations.size()) {
                // considering a new location in the plan
                currentPlan.push(loc);

                // subtract the items available at this location from the remaining items of the order
                int[] newRemaining = arraySubtract(remainingQuantities.peek(), locations.get(loc).quantities);
                remainingQuantities.push(newRemaining);

                if (currentPlan.size() < fulfilmentLimit) {
                    // if we're not at our cap, record how this location affected our remaining items and continue
                    loc++;
                    continue;
                }

                // otherwise, check if we have a complete plan
                if (quantityIsComplete(newRemaining)) {
                    // if the plan completes the order, return it
                    return currentPlan.stream()
                            .map(locations::get)
                            .collect(collectingAndThen(toList(), ImmutableList::copyOf));
                }

                // if we're at the limit of plan sizes, or remaining indexes are greater than slots available
                while (currentPlan.size() == fulfilmentLimit
                        || !currentPlan.isEmpty() && loc + (fulfilmentLimit - currentPlan.size()) >= locations.size()
                ) {
                    // then drop down a level
                    loc = currentPlan.pop();
                    remainingQuantities.pop();
                }
                loc++;
            }
        }

        return NONE;
    }

    private static SourcingPlan buildPlan(List<LocationAndPositions> laps, List<OrderItem> items) {
        int[] quantities = items.stream().mapToInt(OrderItem::getQuantity).toArray();

        ImmutableList.Builder<Fulfilment> fulfilmentBuilder = ImmutableList.builder();
        for (LocationAndPositions lap : laps) {
            ImmutableList.Builder<FulfilmentItem> itemBuilder = ImmutableList.builder();
            for (int i = 0; i < items.size(); i++) {
                if (lap.getQuantities()[i] > 0 && quantities[i] > 0) {
                    final OrderItem orderItem = items.get(i);
                    itemBuilder.add(
                            FulfilmentItem.builder()
                                    .orderItem(orderItem)
                                    .requestedQuantity(Math.min(lap.getQuantities()[i], quantities[i]))
                                    .build()
                    );
                }
            }
            fulfilmentBuilder.add(Fulfilment.builder()
                    .location(lap.getLocation())
                    .items(itemBuilder.build())
                    .build());

            quantities = arraySubtract(quantities, lap.quantities);
        }
        return new SourcingPlan(quantityIsComplete(quantities), fulfilmentBuilder.build());
    }

    /**
     * Container of the information about a single location: location data,
     * inventory quantities in the location, ratings assigned to the location by the Criteria
     */
    @Value
    @AllArgsConstructor
    @Builder
    public static class LocationAndPositions implements Comparable<LocationAndPositions> {
        Location location;
        int[] quantities;
        float[] rating;

        /**
         * Returns whether the location is valid (i.e. is not excluded)
         *
         * @return whether the location is valid (i.e. is not excluded)
         */
        public boolean isValid() {
            if (rating == null) return true;
            for (float c : rating) {
                if (c < 0) return false;
            }
            return true;
        }

        @Override
        public int compareTo(@NotNull LocationAndPositions o) {
            return arrayCompare(this.rating, o.rating);
        }
    }

    private static int[] arraySubtract(int[] a, int... b) {
        int[] result = new int[a.length];
        for (int i = 0; i < a.length; i++) {
            result[i] = a[i] - b[i];
        }
        return result;
    }

    private static int arrayCompare(float[] a, float... b) {
        for (int i = 0; i < a.length; i++) {
            if (a[i] < b[i]) {
                return -1;
            } else if (a[i] > b[i]) {
                return 1;
            }
        }
        return 0;
    }

    private static boolean quantityIsComplete(int... remainingItems) {
        for (int remainingItem : remainingItems) {
            if (remainingItem > 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * Representation of a Sourcing Plan, including the proposed Fulfilments
     */
    @Data
    @AllArgsConstructor
    @NoArgsConstructor
    public static class SourcingPlan {
        boolean fullySourced;
        List<Fulfilment> fulfilments;
    }

    /**
     * Representation of a proposed Fulfilment based on a Location and a list of Items
     */
    @Data
    @Builder
    public static class Fulfilment {
        Location location;
        List<FulfilmentItem> items;
        String type;
    }

    /**
     * Return a list of all outstanding order item quantities, based on the original requested quantities and
     * subtracting all allocated (but non-rejected) quantities on related fulfilments.
     *
     * @param sourcingContext The SourcingContext to check items against (see `loadSourcingContext(context, calculator)`)
     * @return a list of outstanding Items to be used in sourcing logic
     */
    public static List<OrderItem> getUnfulfilledItems(final SourcingContext sourcingContext) {
        if (sourcingContext == null) {
            return ImmutableList.of();
        }

        // collect fulfilment items
        ImmutableList.Builder<FulfilmentItem> fulfilmentItemsBuilder = ImmutableList.builder();
        if (sourcingContext.getFulfilments() != null) {
            for (com.fluentcommerce.util.sourcing.context.model.Fulfilment fulfilment : sourcingContext.getFulfilments()) {
                if (fulfilment.getItems() != null) {
                    fulfilmentItemsBuilder.addAll(fulfilment.getItems());
                }
            }
        }
        final ImmutableList<FulfilmentItem> fulfilmentItems = fulfilmentItemsBuilder.build();

        // calculate unfulfilled items
        final ImmutableList.Builder<OrderItem> unfulfilledItems = ImmutableList.builder();
        if (sourcingContext.getItems() != null) {
            sourcingContext.getItems().stream().forEach(orderItem -> {
                int quantity = orderItem.getQuantity();

                // subtract the quantities already assigned to fulfilments (but not rejected)
                int allocatedQuantity = fulfilmentItems.stream()
                        .filter(fItem -> fItem.getOrderItem().getId().equals(orderItem.getId())) // only items that match this order item
                        .mapToInt(fItem -> fItem.getRequestedQuantity() - fItem.getRejectedQuantity()) // filled is not relevant
                        .sum();

                if (quantity > allocatedQuantity) {
                    unfulfilledItems.add(
                            orderItem.toBuilder()
                                    .quantity(quantity - allocatedQuantity)
                                    .build());
                }
            });
        }

        return unfulfilledItems.build();
    }

    /**
     * Attempts to find a plan for an order based on sourcing strategies defined in the given {@link SourcingProfile}.
     * <p>
     * The method iterates through all {@link SourcingStrategy} entries in the profile and:
     * <ol>
     *   <li>Evaluates whether the strategy's {@link SourcingCondition}s are satisfied for the current {@link SourcingContext}.</li>
     *   <li>Fetches applicable {@link Location}s from the strategy's associated fulfilment network.</li>
     *   <li>Fetches virtual catalogue references and the maximum number of allowed splits (resulting in up to maxSplit + 1 locations)
     *   from the profile and strategy.</li>
     *   <li>Builds the {@link CriteriaArray} used to rank and select locations.</li>
     *   <li>Uses {@link SourcingUtils#findPlanForAllItems} to generate a sourcing plan, taking into account available inventory and ranking rules.</li>
     *   <li>If a fully-sourced plan is found (i.e. all items can be fulfilled), return this plan and exits the loop early.</li>
     * </ol>
     * <p>
     * This method ensures that only one successful sourcing strategy is used, prioritizing the first one that results in a full fulfilment plan.
     *
     * @param context            the rule execution {@link Context}, providing access to APIs and environment variables
     * @param sourcingContext    the {@link SourcingContext} containing unfulfilled items and fulfilment choice
     * @param profile            the {@link SourcingProfile} containing strategies, catalogue, network, and config
     * @param positionStatuses   list of virtual position statuses to be used when loading positions (e.g. "ACTIVE")
     * @param inventoryProcessor optional {@link InventoryProcessor} to post-process inventory before sourcing
     * @return the best {@link SourcingPlan} for an order
     */
    public static SourcingPlan findPlanBasedOnStrategies(
            final Context context,
            final SourcingContext sourcingContext,
            final SourcingProfile profile,
            final List<String> positionStatuses,
            final InventoryProcessor inventoryProcessor
    ) {
        SourcingPlan sourcingPlan = new SourcingPlan();
        for (SourcingStrategy strategy : Objects.requireNonNull(profile.getSourcingStrategies()).stream()
                .filter(sourcingStrategy -> ACTIVE.equals(sourcingStrategy.getStatus()))
                .filter(sourcingStrategy -> evaluateSourcingConditions(sourcingContext, sourcingStrategy.getSourcingConditions()))
                .sorted(Comparator.comparing(SourcingStrategy::getPriority))
                .collect(toList())) {
            final String networkRef = getNetworkRef(profile, strategy);
            final List<Location> allLocations = LocationUtils.getLocationsInNetwork(context, networkRef);
            final String virtualCatalogueRef = getVirtualCatalogueRef(profile, strategy);
            final Integer maxSplit = getMaxSplit(profile, strategy);
            final CriteriaArray criteria = SourcingCriteriaUtils.getCriteria(strategy, SourcingCriteriaUtils.getDefaultCriteria());

            SourcingPlan plan = SourcingUtils.findPlanForAllItems(context,
                    virtualCatalogueRef,
                    sourcingContext,
                    allLocations,
                    criteria,
                    maxSplit,
                    positionStatuses,
                    inventoryProcessor);
            if (plan.isFullySourced()) {
                return plan;
            }
        }
        return sourcingPlan;
    }

    /**
     * Attempts to find a plan for unfulfilled items using fallback sourcing strategies
     * defined in the given {@link SourcingProfile}.
     * <p>
     * This method:
     * <ol>
     *   <li>Iterates through all {@link SourcingStrategy} fallback strategies in the profile.</li>
     *   <li>Applies the first strategy whose sourcing conditions evaluate to {@code true}.</li>
     *   <li>Loads virtual positions for applicable {@link Location}s and products, using virtual catalogues and criteria.</li>
     *   <li>Uses a greedy approach to construct a list of partial {@link SourcingUtils.Fulfilment}s, one per iteration, until:
     *     <ul>
     *       <li>The configured {@code maxSplit} limit is reached (based on strategy config).</li>
     *       <li>All unfulfilled item quantities have been covered.</li>
     *     </ul>
     *   </li>
     *   <li>If at least one fulfilment is generated, return {@link SourcingProfile} with this fulfilments.</li>
     * </ol>
     * <p>
     * Unlike primary sourcing strategies (which require full fulfilment), this fallback method allows generating
     * partial fulfilments as a best-effort mechanism.
     *
     * @param context            the rule execution {@link Context}, providing access to APIs and configuration
     * @param sourcingContext    the {@link SourcingContext} holding the unfulfilled items and fulfilment metadata
     * @param profile            the {@link SourcingProfile} that includes fallback strategies and configuration
     * @param positionStatuses   a list of virtual position statuses to load (e.g. "ACTIVE")
     * @param inventoryProcessor an optional {@link InventoryProcessor} to process positions after loading
     * @return a {@link SourcingPlan} for unfulfilled items
     */
    public static SourcingPlan findPlanBasedOnFallbackStrategies(
            final Context context,
            final SourcingContext sourcingContext,
            final SourcingProfile profile,
            final List<String> positionStatuses,
            final InventoryProcessor inventoryProcessor
    ) {
        SourcingPlan sourcingPlan = new SourcingPlan();
        // only one Sourcing Fallback Strategy can be used to produce partial fulfilments
        Objects.requireNonNull(profile.getSourcingFallbackStrategies()).stream()
                .filter(sourcingStrategy -> ACTIVE.equals(sourcingStrategy.getStatus()))
                .filter(strategy -> evaluateSourcingConditions(sourcingContext, strategy.getSourcingConditions()))
                .sorted(Comparator.comparing(SourcingStrategy::getPriority))
                .findFirst()
                .ifPresent(fallbackStrategy -> {
                    final String networkRef = getNetworkRef(profile, fallbackStrategy);
                    final List<Location> allLocations = LocationUtils.getLocationsInNetwork(context, networkRef);

                    final String virtualCatalogueRef = getVirtualCatalogueRef(profile, fallbackStrategy);
                    final Integer maxSplit = getMaxSplit(profile, fallbackStrategy);

                    final CriteriaArray criteria = SourcingCriteriaUtils.getCriteria(fallbackStrategy, SourcingCriteriaUtils.getDefaultCriteria());

                    final List<SourcingUtils.LocationAndPositions> laps = loadPositions(
                            context,
                            sourcingContext,
                            criteria,
                            virtualCatalogueRef,
                            allLocations,
                            positionStatuses,
                            inventoryProcessor
                    );
                    final List<OrderItem> items = sourcingContext.getUnfulfilledItems();

                    final int fulfilmentLimit = maxSplit
                            - sourcingContext.getFulfilments().size() + 1;

                    final List<SourcingUtils.Fulfilment> fulfilments = new ArrayList<>();
                    for (int i = 0; i < fulfilmentLimit; i++) {
                        final List<OrderItem> remainingItems = itemsMinusFulfilments(items, fulfilments);
                        if (remainingItems.stream().mapToInt(OrderItem::getQuantity).sum() <= 0) break;

                        findHighestValuePartialFulfilment(
                                laps,
                                remainingItems,
                                getLocationRefs(fulfilments)).ifPresent(fulfilments::add);
                    }

                    if (!fulfilments.isEmpty()) {
                        sourcingPlan.setFulfilments(fulfilments);
                    }
                });
        return sourcingPlan;
    }

    /**
     * Build a system rejected {@link Fulfilment} for all unfulfilled {@link OrderItem}s in the given {@link SourcingContext}.
     * <p>
     * This method is typically used as a fallback when no sourcing strategy was able to allocate inventory
     * for the remaining order items. It constructs a {@code Fulfilment} containing all unfulfilled items,
     * marks their {@code rejectedQuantity} as equal to the original item quantity, and associates the fulfilment
     * with a system-defined "rejected" {@link Location}.
     * <p>
     * The rejected fulfilment is then persisted via the standard {@code createFulfilments} mechanism.
     *
     * <p><b>Notes:</b></p>
     * <ul>
     *   <li>If no unfulfilled items are found in the sourcing context, the method exits without side effects.</li>
     *   <li>If fulfilments already exist in the context, their items are scanned to calculate the total
     *       quantity that was previously attempted to be fulfilled for each order item (to populate {@code requestedQuantity}).</li>
     * </ul>
     *
     * @param context                the rule execution {@link Context}, providing access to APIs and configuration
     * @param sourcingContext        the sourcing context containing order items and existing fulfilments
     * @param sysRejectedLocationRef the reference of the system "rejected" location to be used in the rejected fulfilment
     * @return a system rejected {@link Fulfilment} for an order
     */
    public static Fulfilment buildRejectedFulfilment(final Context context,
                                                     final SourcingContext sourcingContext,
                                                     final String sysRejectedLocationRef) {
        final List<OrderItem> unfulfilledItems = sourcingContext.getUnfulfilledItems();
        // if there are no unfulfilled items
        if (unfulfilledItems == null || unfulfilledItems.isEmpty()) {
            return null;
        }
        // collect fulfilment items
        final ImmutableList.Builder<FulfilmentItem> fulfilmentItemsBuilder = ImmutableList.builder();
        if (sourcingContext.getFulfilments() != null) {
            sourcingContext.getFulfilments().stream()
                    .map(com.fluentcommerce.util.sourcing.context.model.Fulfilment::getItems)
                    .filter(Objects::nonNull)
                    .forEach(fulfilmentItemsBuilder::addAll);
        }
        final Location rejectLocation = LocationUtils.getLocationByRef(context, sysRejectedLocationRef);
        final List<FulfilmentItem> items = new ArrayList<>(unfulfilledItems.size());
        unfulfilledItems.forEach(item -> {
            final int quantity = item.getQuantity();
            items.add(FulfilmentItem.builder()
                    .orderItem(item)
                    .requestedQuantity(quantity)
                    .rejectedQuantity(quantity)
                    .build());
        });
        return Fulfilment.builder()
                .location(rejectLocation)
                .items(items)
                .build();
    }

    /**
     * Get either the Virtual Catalogue ref from Sourcing Strategy or default one from Sourcing Profile
     *
     * @param profile  {@link SourcingProfile} to extract default from
     * @param strategy {@link SourcingStrategy} to extract ref from
     * @return Virtual Catalogue ref
     */
    public static String getVirtualCatalogueRef(final SourcingProfile profile,
                                                final SourcingStrategy strategy) {
        final String virtualCatalogueRef = Optional.ofNullable(strategy.getVirtualCatalogue())
                .map(VirtualCatalogue::getRef)
                .orElse(null);
        final String defaultVirtualCatalogueRef = Optional.ofNullable(profile.getDefaultVirtualCatalogue())
                .map(VirtualCatalogue::getRef)
                .orElse(null);
        return ObjectUtils.firstNonNull(virtualCatalogueRef, defaultVirtualCatalogueRef);
    }

    /**
     * Get either the Network ref from Sourcing Strategy or default one from Sourcing Profile
     *
     * @param profile  {@link SourcingProfile} to extract default from
     * @param strategy {@link SourcingStrategy} to extract ref from
     * @return Network ref
     */
    public static String getNetworkRef(final SourcingProfile profile,
                                       final SourcingStrategy strategy) {
        final String networkRef = Optional.ofNullable(strategy.getNetwork())
                .map(Network::getRef)
                .orElse(null);
        final String defaultNetworkRef = Optional.ofNullable(profile.getDefaultNetwork())
                .map(Network::getRef)
                .orElse(null);
        return ObjectUtils.firstNonNull(networkRef, defaultNetworkRef);
    }

    /**
     * Get either the max split configuration from Sourcing Strategy or default one from Sourcing Profile
     *
     * @param profile  {@link SourcingProfile} to extract default from
     * @param strategy {@link SourcingStrategy} to extract configuration from
     * @return max split configuration
     */
    public static Integer getMaxSplit(final SourcingProfile profile,
                                      final SourcingStrategy strategy) {
        Integer maxSplit = strategy.getMaxSplit();
        return maxSplit != null ? maxSplit : profile.getDefaultMaxSplit();
    }

}
