/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;

import com.fluentcommerce.graphql.sourcing.mutation.CreateFulfilmentMutation;
import com.fluentcommerce.graphql.type.AddressId;
import com.fluentcommerce.graphql.type.CreateFulfilmentInput;
import com.fluentcommerce.graphql.type.CreateFulfilmentItemWithFulfilmentInput;
import com.fluentcommerce.graphql.type.OrderId;
import com.fluentcommerce.graphql.type.OrderItemId;
import com.fluentcommerce.types.core.constants.CoreExceptionCode;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.SourcingUtils.Fulfilment;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Fulfilment.FulfilmentItem;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.model.location.LocationType;
import com.fluentretail.rubix.exceptions.RubixException;
import com.fluentretail.rubix.v2.context.Context;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import static com.fluentretail.api.model.order.Order.Type.ClickAndCollect;
import static com.fluentretail.api.model.order.Order.Type.HomeDelivery;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.summingInt;
import static java.util.stream.Collectors.toList;

/**
 * Utility class that provides helper methods for working with Order/Fulfilment entities.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing rule execution.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class OrderUtils {

    /**
     * Subtract the item quantities in a set of fulfilments from a list of order items.
     * This can be used to determine the remaining order items after a set of proposed
     * (but not yet created) fulfilments.
     *
     * @param items       List of unfulfilled items
     * @param fulfilments List of proposed fulfilments (e.g. from
     *                    `SourcingUtils.findHighestValuePartialFulfilment`)
     * @return the initial list of items with updated quantities.
     */
    public static List<OrderItem> itemsMinusFulfilments(final List<OrderItem> items,
                                                        final List<Fulfilment> fulfilments) {
        Map<String, Integer> fulfilmentsTotals = JsonUtils.stream(fulfilments)
                .flatMap(fulfilment -> fulfilment.getItems().stream())
                .collect(groupingBy(item -> item.getOrderItem().getProduct().getRef(), summingInt(FulfilmentItem::getRequestedQuantity)));

        return items.stream()
                .map(i -> i.toBuilder()
                        .quantity(i.getQuantity() - fulfilmentsTotals.getOrDefault(i.getProduct().getRef(), 0))
                        .build()
                )
                .collect(toList());
    }

    /**
     * Fills set of fulfilments with types.
     *
     * @param sourcingContext context of the current sourcing request
     * @param fulfilments     list of fulfilments to fill in types
     */
    public static void fillFulfilmentType(final SourcingContext sourcingContext, List<Fulfilment> fulfilments) {
        final FulfilmentChoice fulfilmentChoice = sourcingContext.getFulfilmentChoice();
        for (Fulfilment fulfilment : fulfilments) {
            final String fulfilmentType = getFulfilmentType(fulfilmentChoice, fulfilment.getLocation().getType());
            fulfilment.setType(fulfilmentType);
        }
    }

    /**
     * Create a set of fulfilments against the current sourcing request (i.e. Order).
     *
     * @param context         rule context object
     * @param sourcingContext context of the current sourcing request
     * @param fulfilments     list of fulfilments to create
     */
    public static void createFulfilments(final Context context,
                                         final SourcingContext sourcingContext,
                                         final List<Fulfilment> fulfilments) {
        final FulfilmentChoice fulfilmentChoice = sourcingContext.getFulfilmentChoice();
        for (Fulfilment fulfilment : fulfilments) {
            final String toAddressId = fulfilmentChoice.getAddress().getId();
            final CreateFulfilmentInput createFulfilmentInput = CreateFulfilmentInput.builder()
                    .fromAddress(AddressId.builder()
                            .id(fulfilment.getLocation().getPrimaryAddress().getId())
                            .build())
                    .toAddress(AddressId.builder()
                            .id(toAddressId)
                            .build())
                    .type(fulfilment.getType())
                    .items(mapItemsToInput(fulfilment.getItems()))
                    .order(OrderId.builder()
                            .id(sourcingContext.getId())
                            .build())
                    .ref(UUID.randomUUID().toString())
                    .deliveryType(sourcingContext.getFulfilmentChoice().getDeliveryType())
                    .fulfilmentChoiceRef(sourcingContext.getFulfilmentChoice().getRef())
                    .build();
            final CreateFulfilmentMutation createFulfilment = CreateFulfilmentMutation.builder()
                    .input(createFulfilmentInput)
                    .build();
            context.action().mutation(createFulfilment);
        }
    }

    /**
     * Determine Fulfilment type based on provided Fulfilment Choice and Location types.
     *
     * @param fulfilmentChoice fulfilment choice used for the fulfilment
     * @param locationType     type of the location for the fulfilment
     * @return fulfilment type
     */
    public static String getFulfilmentType(final FulfilmentChoice fulfilmentChoice, final String locationType) {
        String fulfilmentType;
        switch (fulfilmentChoice.getType()) {
            case ClickAndCollect:
            case HomeDelivery:
                fulfilmentType = fulfilmentChoice.getType() + "_" + getSuffixAsPerLocationType(locationType);
                break;
            default:
                throw new RubixException(CoreExceptionCode.ErrorCode.ERROR_CODE_400.getValue(), "{{fulfilmentType}} parameter to be set in the order workflow due to " + fulfilmentChoice.getType() + " order type");
        }
        return fulfilmentType;
    }

    private static String getSuffixAsPerLocationType(@NotNull final String locationType) throws RubixException {
        try {
            return LocationType.valueOf(locationType.toUpperCase(Locale.getDefault())).getValue();
        } catch (final IllegalArgumentException e) {
            throw new RubixException(CoreExceptionCode.ErrorCode.ERROR_CODE_400.getValue(), "Invalid location type: " + locationType, e);
        }
    }

    private static List<CreateFulfilmentItemWithFulfilmentInput> mapItemsToInput(final List<FulfilmentItem> items) {
        final List<CreateFulfilmentItemWithFulfilmentInput> mappedItems = new ArrayList<>();
        for (final FulfilmentItem item : items) {
            mappedItems.add(CreateFulfilmentItemWithFulfilmentInput.builder()
                    .ref(item.getOrderItem().getProduct().getRef())
                    .orderItem(OrderItemId.builder()
                            .id(item.getOrderItem().getId())
                            .build())
                    .requestedQuantity(Optional.ofNullable(item.getRequestedQuantity())
                            .orElse(0))
                    .rejectedQuantity(Optional.ofNullable(item.getRejectedQuantity())
                            .orElse(0))
                    .build());
        }
        return mappedItems;
    }
}
