# Sourcing Module #

This module contains utility functions to simplify the consistent application of sourcing logic to Order and Fulfilment
Options APIs.

### What is this repository for? ###

### How do I get set up? ###

Build the jar with `mvn clean package` and find at `util/sourcing/target/util-sourcing-1.0.0.jar`

### Using the jar ###

The best way is to install the jar into your local maven repo and then use it as a normal maven dependency.

```bash
mvn install:install-file \
   -Dfile=<path-to-file> \
   -DgroupId=com.fluentcommerce \
   -DartifactId=util-sourcing \
   -Dversion=1.0.0 \
   -Dpackaging=jar \
   -DgeneratePom=true
```

To compile this project, we can do the same with the required `util-test` package.

```bash
mvn install:install-file \
   -Dfile=<path-to-file> \
   -DgroupId=com.fluentcommerce \
   -DartifactId=util-test \
   -Dversion=1.2.3 \
   -Dpackaging=jar \
   -DgeneratePom=true
```

### Developer Usage ###

These utilities are designed to simplify the creation of rules that contribute to order sourcing - that is, assigning
items to locations by creating fulfilments.

The intent is to make it simpler to build up a collection of sourcing strategy rules that can be combined in workflow
to produce an overall sourcing strategy for the order.

### SourcingUtils.findPlanBasedOnStrategies ###

Finds the best sourcing plan for an order based on sourcing strategies defined in a sourcing profile.

This method iterates over all sourcing strategies in the profile,
evaluates conditions for each, and returns the best sourcing plan found.
It stops at the first successful strategy that fully fulfills the order.

#### What it does:

1. Iterates over each `SourcingStrategy` in the provided `SourcingProfile`.
2. Sorting the strategies by their priority in ascending order and filtering out any strategies that are not ACTIVE.
3. Evaluates the sourcing conditions of the strategy against the current order context.
4. If conditions are met, retrieves:
    - Relevant network reference and locations
    - Virtual catalogue applicable for the strategy
    - Maximum allowed splits (`maxSplit`)
    - Sourcing criteria to prioritize locations
5. Calls `findPlanForAllItems(...)` to generate a sourcing plan based on the above parameters.
6. If a plan fully sources all items, returns this plan.
7. Stops processing further strategies once a full fulfilment plan is created.

#### Input:

- `Context context`: Execution environment
- `SourcingContext sourcingContext`: Contains order and unfulfilled items
- `SourcingProfile profile`: Contains sourcing strategies and related configuration
- `List<String> positionStatuses`: Virtual position statuses to consider (e.g., ACTIVE)
- `InventoryProcessor inventoryProcessor`: Functional interface to transform virtual positions before sourcing

#### Output:

- Returns a `SourcingPlan` object containing:

- The list of the proposed `Fulfilment`s

If no valid combination is found, returns null.

#### Notes:

- Strategies are evaluated sequentially; the first one that produces a full fulfilment plan is used.
- Uses `SourcingUtils.findPlanForAllItems(...)` internally to generate sourcing plans.
- Ensures fulfilments comply with strategy-specific conditions and criteria.
- You can customize inventory before sourcing using `InventoryProcessor`.

#### Usage Example of `SourcingUtils.findPlanBasedOnStrategies`

```java
// Load the sourcing profile for the current context
SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);

// Check that sourcing strategies are available
if (sourcingProfile.getSourcingStrategies() == null) {
        return; // Exit if no strategies found
}

// Load sourcing context with unfulfilled order items
SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(
        context,
        SourcingUtils::getUnfulfilledItems
);

// Define custom InventoryProcessor (optional)
InventoryProcessor processor = (ctx, positions) -> positions.stream()

        // exclude certain virtual positions
        .filter(position -> {
            // ... skip position for some reason

            return true;
        })
        // transform remaining positions
        .map(position -> {
            // ... adjust quantity for some reason
            return position;
        })
        .collect(Collectors.toList());

// Trigger finding a plan based on sourcing strategies
SourcingPlan plan = SourcingUtils.findPlanBasedOnStrategies(
        context,
        sourcingContext,
        sourcingProfile,
        ImmutableList.of(Constants.Status.ACTIVE),  // Consider only positions with ACTIVE status
        processor                                   // Optional inventory transformer
);
// Check that the plan contains the fulfilments
if (plan.getFulfilments() != null) {
        fillFulfilmentType(sourcingContext, plan.getFulfilments());  // Set the fulfilment type to fulfilments
        createFulfilments(context, sourcingContext, plan.getFulfilments());  // Trigger the fulfilments creation
};
```
For additional examples and tests, see 
`SourcingUtilsRuleTest.should_create_single_fulfilment_when_running_basic_sourcing_rule` and
`SourcingUtilsRuleTest.should_create_multiple_fulfilments_when_running_basic_sourcing_rule`.

### SourcingUtils.findPlanBasedOnFallbackStrategies ###

Finds the sourcing plan for unfulfilled items using fallback sourcing strategies when no primary strategy fully satisfies the order.

This method attempts to apply the first fallback strategy that matches the sourcing context conditions and generates one
or more partial fulfilments using the best available locations.

#### What it does:

1. Iterates over each `SourcingFallbackStrategy` in the provided `SourcingProfile`. 
2. Sorting the strategies by their priority in ascending order and filtering out any strategies that are not ACTIVE.
3. Evaluates sourcing conditions for each strategy against the current order context.
4. If a strategy is applicable:
    - Retrieves relevant network reference and locations
    - Fetches virtual catalogue applicable for the strategy
    - Reads maximum allowed splits (`maxSplit`)
    - Obtains sourcing criteria to evaluate location suitability
5. Loads virtual positions (`LocationAndPositions`) based on unfulfilled order items and filtered by `positionStatuses`
   and processed by `inventoryProcessor`.
6. Calculates a fulfilment limit based on `maxSplit` and number of existing fulfilments.
7. Iteratively tries to find the most valuable partial fulfilments until all items are covered or limit is reached.
8. Returns a valid generated partial plan.

#### Input:

- `Context context`: Execution environment
- `SourcingContext sourcingContext`: Contains order and unfulfilled items
- `SourcingProfile profile`: Contains fallback sourcing strategies and configuration
- `List<String> positionStatuses`: Virtual position statuses to consider (e.g., ACTIVE)
- `InventoryProcessor inventoryProcessor`: Functional interface to process raw virtual positions before sourcing

#### Output:

- Returns a `SourcingPlan` object containing:

- The list of the partial `Fulfilment`s

If no valid combination is found, returns null.

#### Notes:

- Only **one fallback strategy** is used — the first one that satisfies sourcing conditions.
- Supports **partial sourcing** — fulfilments may not fully satisfy the order.

#### Usage Example of `SourcingUtils.findPlanBasedOnFallbackStrategies`

```java
// Load the sourcing profile for the current context
SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);

// Check that fallback strategies are available
if (sourcingProfile.getSourcingFallbackStrategies() == null) {
    return; // Exit if no fallback strategies found
}

// Load sourcing context with unfulfilled order items
SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(
        context,
        SourcingUtils::getUnfulfilledItems
);

// Define custom InventoryProcessor (optional)
InventoryProcessor customProcessor = ...

// Trigger finding a plan based on fallback sourcing strategies
SourcingPlan plan = SourcingUtils.findPlanBasedOnFallbackStrategies(
    context,
    sourcingContext,
    sourcingProfile,
    ImmutableList.of(Constants.Status.ACTIVE), // Consider only ACTIVE inventory positions
    customProcessor                            // Apply custom inventory transformation if needed
);
// Check that the plan contains the fulfilments
if (plan.getFulfilments() != null) {
        fillFulfilmentType(sourcingContext, plan.getFulfilments());  // Set the fulfilment type to fulfilments
        createFulfilments(context, sourcingContext, plan.getFulfilments());  // Trigger the fulfilments creation
};
```
For additional examples and tests, see
`SourcingUtilsRuleTest.should_create_single_fulfilment_when_running_fallback_sourcing_rule`.

### SourcingUtils.buildRejectedFulfilment ###

Builds a **rejected fulfilment** for all remaining unfulfilled items in the sourcing context.  
Typically used to **mark unfulfillable items** by assigning them to a system-level "rejected" location.

#### What it does:

1. Checks if there are any unfulfilled items in the `SourcingContext`.
2. Collects all previously assigned `FulfilmentItem`s (if any).
3. Loads a special "rejection" location using the given location ref.
4. Builds a new fulfilment:
    - Adds all unfulfilled items.
    - Marks them as fully rejected (`rejectedQuantity = orderItem.getQuantity()`).
    - Includes already requested quantity if the item was partially fulfilled.
5. Returns the built rejected fulfillment for further processing.

#### Input:

- `Context context`: Execution context
- `SourcingContext sourcingContext`: Sourcing state including unfulfilled items
- `String sysRejectedLocationRef`: Reference to the system "rejected" location

#### Output:

- Returns the rejected `Fulfilment`

If no valid filfilment is found, returns null.

#### Notes:

- This method **preserves partially assigned quantities**, if the item has already been part of another fulfilment.
- Marks remaining quantity as rejected.
- Ensures that all items are included, even if partially fulfilled elsewhere.

#### Usage Example of `SourcingUtils.buildRejectedFulfilment`

```java

// Load sourcing context with unfulfilled items
SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(
        context,
        SourcingUtils::getUnfulfilledItems
);

// Specify reference to the system rejection location
String rejectedLocationRef = context.getProp(Constants.PropertyField.PROP_SYSTEM_REJECTED_LOC_REF);

// Build a rejected fulfilment for the remaining items
Fulfilment rejectedFulfilment = SourcingUtils.buildRejectedFulfilment(context, sourcingContext, rejectedLocationRef);
// Check that the rejected fulfilment is builded
if (rejectedFulfilment != null) {
        fillFulfilmentType(sourcingContext, Collections.singletonList(rejectedFulfilment)); // Set the fulfilment type to fulfilments
        createFulfilments(context, sourcingContext, Collections.singletonList(rejectedFulfilment)); // Trigger the fulfilments creation
}
```
For additional examples and tests, see 
`SourcingUtilsRuleTest.should_not_create_rejected_fulfilment_when_no_unfulfilled_order_items`.

### Inventory Processor ###

The InventoryProcessor is a functional utility used in sourcing to preprocess inventory data (VirtualPosition objects)
before they are evaluated for inclusion in a sourcing plan.

It is passed to sourcing utilities such as `findPlanBasedOnStrategies(...)`,
`findPlanBasedOnFallbackStrategies(...)`.

An InventoryProcessor receives the full list of virtual positions and can:

- Filter out certain positions (e.g. not available, damaged, restricted stock)
- Modify the quantity or attributes of positions

The InventoryProcessor runs after inventory is fetched from the virtual catalogue but before any plans are generated.
This gives you a precise opportunity to shape the inventory input used in sourcing strategies.

Finally, the simplest way to add your own inventory processor is to provide a lambda function.

```java
InventoryProcessor inventoryProcessor = (sourcingContext, positions) -> {
    int requestedQuantity = sourcingContext.getUnfulfilledItems().stream()
            .filter(item -> PRODUCT1.equals(item.getProduct().getRef()))
            .mapToInt(OrderItem::getQuantity)
            .sum();

    if (requestedQuantity >= 100) {
        return positions;
    }

    return positions.stream()
            .map(position -> {
                if (PRODUCT1.equals(position.getProductRef())) {
                    return position.toBuilder()
                            .quantity(0)
                            .build();
                } else {
                    return position;
                }
            })
            .collect(Collectors.toList());
};
```

### Unfulfilled Item Processor ###

The UnfulfilledItemProcessor is a functional interface designed to extract the list of order items that still require
sourcing.

This processor is a plug-and-play abstraction around the logic for computing unfulfilled items from the SourcingContext.
It is especially useful when customizing sourcing workflows or swapping in alternative definitions of "unfulfilled"
(e.g. based on business rules ).
Note: It is possible that order items from different brands need to be fulfilled separately, 
potentially in different sourcing rules or workflows, to respect brand-specific fulfillment requirements.

#### Default Implementation

A common implementation uses the standard utility SourcingUtils.getUnfulfilledItems(...), which compares original
OrderItem quantities to the quantity already allocated to fulfilments.

```java 
 final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
```

### SourcingUtils.findPlanForAllItems ###

Builds a sourcing plan that fulfills all unfulfilled items in an order using one or more available locations.

This method evaluates stock availability per location (via `loadPositions`) and determines the optimal
combination of locations that can collectively fulfill the required quantities, while respecting a limit on the number
of splits.

#### What it does:

1. Extracts unfulfilled items from the `SourcingContext`.
2. Returns an empty plan immediately if the item list is `null` or empty.
3. Computes required quantities for all unfulfilled items.
4. Calls `loadPositions(...)` to retrieve stock availability per location and product.
5. Attempts to find the best combination of `LocationAndPositions` that fully satisfies the required quantities using *
   *up to `maxSplit + 1` locations**.
6. Converts the chosen combination into a final `SourcingPlan`.

#### Input:

- `Context context`: Execution environment
- `String catalogueRef`: Virtual catalogue reference to search stock in
- `SourcingContext sourcingContext`: Contains the unfulfilled order items
- `List<Location> locations`: List of potential source locations
- `CriteriaArray sourcingCriteria`: Determines location ranking
- `int maxSplit`: Maximum number of splits allowed in the plan — results in up to (maxSplit + 1) locations
- `List<String> positionStatuses`: List of acceptable inventory statuses (e.g., ACTIVE)
- `InventoryProcessor inventoryProcessor` – custom processor for transforming inventory data before sourcing

#### Output:

Returns a `SourcingPlan` object containing:

- A selected combination of `LocationAndPositions` that collectively fulfill all required item quantities
- The list of fulfilled `OrderItem`s

If no valid combination is found, returns an empty plan.

#### Notes:

- Relies on `loadPositions(...)` to evaluate inventory per location.
- Stops early once a valid plan is found (no exhaustive search).
- Optimized for performance in large fulfillment networks.

#### Usage Example of `SourcingUtils.findPlanForAllItems`

This example demonstrates how to evaluate multiple sourcing strategies and generate a sourcing plan for unfulfilled
items.  
The first strategy that meets the conditions and results in a fully sourced plan can be used to trigger fulfilment
creation.

```java
// Load the sourcing profile from the context
SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);

// Load all unfulfilled order items into a sourcing context
SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(
        context,
        SourcingUtils::getUnfulfilledItems
);

// Iterate through sourcing strategies in the profile
for (SourcingStrategy strategy : Objects.requireNonNull(sourcingProfile.getSourcingStrategies())) {

    // Skip strategies whose conditions are not satisfied
    if (!evaluateSourcingConditions(sourcingContext, strategy.getSourcingConditions())) {
        continue;
    }

    // Collect sourcing parameters from strategy and profile
    String networkRef = SourcingUtils.getNetworkRef(sourcingProfile, strategy);
    List<Location> allLocations = LocationUtils.getLocationsInNetwork(context, networkRef);
    String virtualCatalogueRef = SourcingUtils.getVirtualCatalogueRef(sourcingProfile, strategy);
    Integer maxSplit = SourcingUtils.getMaxSplit(sourcingProfile, strategy);
    CriteriaArray criteria = SourcingCriteriaUtils.getCriteria(strategy, SourcingCriteriaUtils.getDefaultCriteria());

    // Generate a sourcing plan based on active inventory and criteria
    SourcingPlan plan = SourcingUtils.findPlanForAllItems(
            context,
            virtualCatalogueRef,
            sourcingContext,
            allLocations,
            criteria,
            maxSplit,
            ImmutableList.of("ACTIVE"), // Consider only ACTIVE inventory positions
            null
    );

    // If the plan is fully sourced, proceed to fulfilment
    if (plan.isFullySourced()) {
        // e.g., createFulfilments(context, sourcingContext, plan.getFulfilments());
        break;
    }
}
```

### SourcingUtils.loadPositions ###

Fetches and prepares a prioritized list of locations with stock availability per order item.

This method is used during the sourcing process to evaluate which locations have the required
inventory for a given order and how suitable they are, based on sourcing criteria.

#### What it does:

1. Sorts and filters candidate locations using the provided `SourcingCriteria`.
2. Limits the number of locations to avoid exceeding the virtual position cap (`MAX_POSITIONS`).
3. Sends a GraphQL query (`GetVirtualPositionsQuery`) to retrieve available virtual positions per location and product.
4. Groups inventory results into a fast-access structure for mapping location and item quantities.
5. Returns a list of `LocationAndPositions` objects:
    - `Location` object
    - An integer array representing available quantities for each unfulfilled item
    - Ratings for that location

#### Input:

- `Context context`: Execution environment
- `SourcingContext sourcingContext`: Contains the unfulfilled order items
- `CriteriaArray sourcingCriteria`: Determines location ranking
- `String catalogueRef`: Virtual catalogue reference to search stock in
- `List<Location> locations`: List of potential source locations
- `List<String> positionStatuses`: List of acceptable inventory statuses (e.g., ACTIVE)

#### Output:

Returns a sorted, list of `LocationAndPositions`, each entry containing:

- Location reference
- Array of stock quantities for each item
- Computed rating (based on sourcing criteria)

```json
[
   {
      "location": "location1",
      "quantities": [3, 5],
      "rating": [0.9, 0.8]
   },
   {
      "location": "location2",
      "quantities": [1, 2],
      "rating": [0.7, 0.6]
   },
   {
      "location": "location3",
      "quantities": [2, 2],
      "rating": [0.85, 0.75]
   }
]
```

#### Notes:

- Only unfulfilled items are considered.
- The method prioritizes performance by capping the number of queried positions.

#### Usage Example of `SourcingUtils.loadPositions`

This example demonstrates how to load available virtual positions for a set of candidate locations and unfulfilled
order items.  
The `loadPositions` method returns a list of locations with virtual stock data mapped to order items and ranked by
sourcing criteria.

```java
// Load sourcing profile and sourcing context
SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);
SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(
        context,
        SourcingUtils::getUnfulfilledItems
);

// Choose a sourcing strategy from the profile
SourcingStrategy sourcingStrategy = ...; // e.g., sourcingProfile.getSourcingStrategies().get(0)

// Extract sourcing parameters
String networkRef = SourcingUtils.getNetworkRef(sourcingProfile, sourcingStrategy);
List<Location> locations = LocationUtils.getLocationsInNetwork(context, networkRef);
String catalogueRef = SourcingUtils.getVirtualCatalogueRef(sourcingProfile, sourcingStrategy);
CriteriaArray sourcingCriteria = SourcingCriteriaUtils.getCriteria(sourcingStrategy);

// Load inventory positions grouped by location and item
List<SourcingUtils.LocationAndPositions> locationAndPositionsList = SourcingUtils.loadPositions(
        context,
        sourcingContext,
        sourcingCriteria,
        catalogueRef,
        locations,
        ImmutableList.of("ACTIVE"), // Only consider ACTIVE virtual positions
        null
);
```

### SourcingUtils.searchPermutationsForAllMatchPlan ###

Finds the minimal combination of locations that can fully satisfy the required item quantities, within a maximum allowed
number of splits.

This method iteratively checks all valid combinations of locations (single, pair, triplet, etc.) up to a given `maxSplit`
limit.
It selects the first group of locations whose combined inventory covers all required quantities for the order.

#### What it does:

1. Iterates over all location group sizes from `1` to `maxSplit + 1`.
2. For each group size:
    - Recursively builds permutations of locations.
    - After each location is added to the plan, subtracts its quantities from the remaining required quantities.
    - Tracks remaining quantities on a stack to efficiently backtrack.
3. If a combination satisfies all quantities (`quantityIsComplete`), returns the plan immediately.
4. If no combination is found, returns an empty result.

#### Input:

- `List<LocationAndPositions> locations`:  
  A list of locations, each with associated available quantities per item.

- `int maxSplit`:  
  The maximum number of splits allowed to fulfill the order.

- `int... quantities`:  
  Required quantities for each item in the order.

```json
{
   "locations": [
      {
         "location": "location1",
         "quantities": [3, 5],
         "rating": [0.9, 0.8]
      },
      {
         "location": "location2",
         "quantities": [1, 2],
         "rating": [0.7, 0.6]
      },
      {
         "location": "location3",
         "quantities": [2, 2],
         "rating": [0.85, 0.75]
      }
   ],
   "maxSplit": 2,
   "orderQuantities": [4, 5]
}
```

#### Output:

- `List<LocationAndPositions>`:  
  The first valid combination of locations that together fulfill the order within the allowed number of splits.  
  Returns an empty list if no valid combination is found.

```json
[
   {
      "location": "location1",
      "quantities": [3, 5],
      "rating": [0.9, 0.8]
   },
   {
      "location": "location3",
      "quantities": [2, 2],
      "rating": [0.85, 0.75]
   }
]
```

#### Usage Example of `SourcingUtils.searchPermutationsForAllMatchPlan`

This example shows how to prepare order item quantities and find the best combination of locations and positions to
fulfil the order, respecting the maximum allowed splits.

```java
final int maxSplit = ...;
final List<SourcingUtils.LocationAndPositions> locationAndPositionsList = ...;
final List<OrderItem> items = sourcingContext.getUnfulfilledItems();

final int[] quantities = items.stream()
        .mapToInt(OrderItem::getQuantity)
        .toArray();

// now we can try to find the best combo of laps to fill the order
final List<SourcingUtils.LocationAndPositions> bestCombo =
        searchPermutationsForAllMatchPlan(locationAndPositionsList, maxSplit, quantities);
```
### SourcingUtils.findHighestValuePartialFulfilment ###

Sometimes we can't fulfil an entire order — whether due to stock discrepancies, high concurrency, or business
constraints (like fulfilment split limits).
In such cases, we may need to attempt partial fulfilment — ideally as a fallback when primary strategies fail.
The findHighestValuePartialFulfilment utility takes a list of pre-evaluated locations (with associated items and a
rating), and tries to select the single best fulfilment that maximizes value — as determined by the precomputed rating.

It is intended to be used in a loop to create multiple partial fulfilments,
helping get as close as possible to fulfilling the entire order, while avoiding already-used locations.

The utility does the following:
Filters out locations already used in previous fulfilments (excludedLocations).
Selects the location with the highest rating that can fulfil at least part of the remaining items.

Builds a fulfilment plan using only that location and returns the resulting Fulfilment.

### Extending the Engine ###

Fluent's sourcing logic heavily relies on three core methods from SourcingUtils:
`SourcingUtils.findPlanBasedOnStrategies`,
`SourcingUtils.findPlanBasedOnFallbackStrategies`,
`SourcingUtils.buildRejectedFulfilment`

These methods are responsible for building fulfilment plan based on loaded default or custom-defined sourcing context,
as well as default or custom-defined criteria and conditions.
Criteria and conditions are used to score, filter, and exclude locations during sourcing evaluation.

Moreover, the framework is extensible — you can register your own custom SourcingContext, SourcingContextLoader,
SourcingCriterion or SourcingCondition and SourcingConditionOperator to plug into the strategy evaluation process.

The sourcing context (e.g., an Order, a FulfilmentChoice, or a custom entity) is loaded via a pluggable
SourcingContextLoader and provides the necessary data for evaluating sourcing strategies.

Then create and register a custom loader like this if needed

```java
public class CustomContext implements SourcingContext {
    ...
}

public class CustomContextLoader implements SourcingContextLoader {

    @Override
    public SourcingContext load(Context context, UnfulfilledItemProcessor processor) {
        Event event = context.getEvent();
        // Map raw event data to custom context
        CustomContext customContext = new CustomContext();
        // fill the sourcing custom context
        ...
        return customContext;
    }
}

static {
    SourcingContextUtils.registerContextLoader("custom", new CustomContextLoader());
}
```

Create and register a custom condition if needed

```java
public class CustomCondition implements SourcingCondition {

    @Override
    public boolean evaluateWithContext(JsonNode sourcingContext) {
        //get data from the context and evaluate it
        ...
    }
}

static {
    SourcingConditionTypeRegistry.register("company.custom.condition", new CustomCondition());
}
```

Create and register a custom condition operator if needed

```java
public class StartsWithOperator implements SourcingConditionOperator {

    @Override
    public boolean evaluate(Object... operands) {
        ...
    }
}

static {
    SourcingConditionOperatorRegistry.register("starts_with", new StartsWithOperator());
}
```

Create and register a custom criterion if needed

```java
public class CustomCriterion implements SourcingCriterion {

    @Override
    public float apply(SourcingCriteriaUtils.CriterionContext criterionContext) {
        // Example: return static rating
        return 0.75f;
    }

    @Override
    public void parseParams(JsonNode params) {
        // Optional: parse criterion configuration from strategy JSON
    }

    @Override
    public float normalize(float min, float max, float rating) {
        // Optional: use default normalization or override
        return SourcingCriterion.super.normalize(min, max, rating);
    }
}

static {
    SourcingCriteriaTypeRegistry.register("company.custom.criterion", CustomCriterion.class);
}
```

### Auto-Generation of Sourcing Criterion Setting

#### Overview

This module provides an automatic mechanism for generating a **Sourcing Criteria Setting**  in JSON format of all sourcing criteria using custom Java annotations.

At build time, the generator writes json file to **target/**  
When run via `java -jar`, it writes the file alongside the JAR.  
The JSON can be consumed by frontend tools, configuration UIs, or exported for documentation.

---

#### How It Works

- The generator scans all classes registered in `SourcingCriteriaTypeRegistry`.
- It looks for the `@SourcingCriterionInfo` annotation on each class.
- It collects all fields annotated with `@SourcingCriterionParam`.
- It builds a clean, structured JSON document representing all available sourcing criteria and their parameter definitions.
- The file is saved during Maven build.

---

#### Usage

 You can invoke the generator if you run the following command:  
   ```bash
   java -jar <path>util-sourcing-<version>-sources.jar
   ```
 Replace **\<version>** with the version from your pom.xml or the actual built JAR name  

 Replace **\<path>** with the path to your JAR file  

 **Example:**
   ```bash
   java -jar util/sourcing/target/util-sourcing-2.1.0-sources.jar
   ```
#### Output
The generated file will be stored at the same folder where jar is located.

#### How to Extend
Add a New Sourcing Criterion:
1. Create a class and annotate it with **@SourcingCriterionInfo**.
2. Define fields and annotate them with **@SourcingCriterionParam**.
3. Register the class in **SourcingCriteriaTypeRegistry**.

The generator will automatically pick it up and include it in the output.

Add Select Component Options to Params if needed:
- If a parameter uses a select component, use **@SourcingCriterionParamSelectComponentOption** to define available options with labels and values.

**Example**:

```java
@SourcingCriterionParam(
    name = "valueUnit",
    component = "select",
    selectComponentOptions = {
        @SourcingCriterionParamSelectComponentOption(
            label = "Kilometres",
            value = "kilometres"
        ),
        @SourcingCriterionParamSelectComponentOption(
            label = "Miles",
            value = "miles"
        )
    }
)
private String unit;
```

### Who do I talk to? ###

* Ben Harrison (author)
* Jaime Madrid (support)