/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fluentcommerce.util.sourcing.SourcingUtils;
import com.fluentcommerce.util.sourcing.context.DefaultSourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentretail.api.model.attribute.Attribute;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.assertEquals;

class SourcingCriteriaTest {
    private static final ObjectMapper mapper = new ObjectMapper();

    private final static Location LOCATION = Location.builder()
            .primaryAddress(Address.builder()
                    .longitude(0d)
                    .latitude(0d)
                    .build())
            .attributes(Collections.singletonList(Attribute.builder()
                    .name("CURRENT_DAILY_ORDER_CAPACITY")
                    .type("INTEGER")
                    .value(100)
                    .build()))
            .build();
    
    private final static String VALUE = "value";

    private static Stream<Arguments> distanceArguments() {
        return Stream.of(
                Arguments.of(0, 0, 0f),
                Arguments.of(0, 10, 1111.9492f)
        );
    }

    @ParameterizedTest
    @MethodSource("distanceArguments")
    void should_return_expected_rating_when_distance_criterion_is_applied(final double lon, final double lat, final float expected) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .fulfilmentChoice(FulfilmentChoice.builder()
                                .address(Address.builder()
                                        .longitude(lon)
                                        .latitude(lat)
                                        .build())
                                .build())
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationDistanceCriterion();
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("distanceExclusionKilometresArguments")
    void should_return_expected_rating_when_distance_exclusion_criterion_is_applied_for_kilometres(final double lon, final double lat, final float expected) {

        double rating = getDistanceExclusionCriterionRating(lon, lat,100, null);

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("distanceExclusionMilesArguments")
    void should_return_expected_rating_when_distance_exclusion_criterion_is_applied_for_miles(final double lon, final double lat, final float expected) {

        double rating = getDistanceExclusionCriterionRating(lon, lat, 100, "MILES");

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("distanceExclusionUnsupportedUnitArguments")
    void should_return_expected_rating_when_distance_exclusion_criterion_is_applied_for_unsupported_unit(final double lon, final double lat, final float expected) {

        double rating = getDistanceExclusionCriterionRating(lon, lat, 100, "METRES");

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("bandedDistanceKilometresArguments")
    void should_return_expected_rating_when_banded_distance_criterion_is_applied_for_kilometres(final double lon, final double lat, final float expected) {

        double rating = getDistanceBandedCriterionRating(lon, lat, ImmutableList.of(0, 100, 200, 300), null);

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("bandedDistanceMilesArguments")
    void should_return_expected_rating_when_banded_distance_criterion_is_applied_for_miles(final double lon, final double lat, final float expected) {

        double rating = getDistanceBandedCriterionRating(lon, lat, ImmutableList.of(0, 50, 100, 150), "MILES");

        assertEquals(expected, rating, 0.001f);
    }

    @ParameterizedTest
    @MethodSource("bandedDistanceUnsupportedUnitArguments")
    void should_return_expected_rating_when_banded_distance_criterion_is_applied_for_unsupported_unit(final double lon, final double lat, final float expected) {

        double rating = getDistanceBandedCriterionRating(lon, lat, ImmutableList.of(0, 100, 200, 300), "METRES");

        assertEquals(expected, rating, 0.001f);
    }

    private double getDistanceExclusionCriterionRating(final double lon, final double lat,
                                                       final double maxDistance, final String unit) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .fulfilmentChoice(FulfilmentChoice.builder()
                                .address(Address.builder()
                                        .longitude(lon)
                                        .latitude(lat)
                                        .build())
                                .build())
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationDistanceExclusionCriterion();
        
        ImmutableMap.Builder<String, Object> paramsBuilder = ImmutableMap.builder();
        paramsBuilder.put(VALUE, maxDistance);
        if (unit != null) {
            paramsBuilder.put("valueUnit", unit);
        }
        
        criterion.parseParams(mapper.valueToTree(paramsBuilder.build()));
        return criterion.apply(context);
    }

    private double getDistanceBandedCriterionRating(final double lon, final double lat,
                                                    final List<Integer> bands, final String unit) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .fulfilmentChoice(FulfilmentChoice.builder()
                                .address(Address.builder()
                                        .longitude(lon)
                                        .latitude(lat)
                                        .build())
                                .build())
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationDistanceBandedCriterion();
        
        ImmutableMap.Builder<String, Object> paramsBuilder = ImmutableMap.builder();
        paramsBuilder.put(VALUE, bands);
        if (unit != null) {
            paramsBuilder.put("valueUnit", unit);
        }
        
        criterion.parseParams(mapper.valueToTree(paramsBuilder.build()));
        return criterion.apply(context);
    }

    private static Stream<Arguments> distanceExclusionKilometresArguments() {
        return Stream.of(
                Arguments.of(0, 0, 1f),     // 0 км
                Arguments.of(0, 0.945, -1f) // ~ 105 км
        );
    }

    private static Stream<Arguments> distanceExclusionMilesArguments() {
        return Stream.of(
                Arguments.of(0, 0, 1f),      // 0 miles
                Arguments.of(0, 1.521, -1f)  // ~ 105 miles
        );
    }

    private static Stream<Arguments> distanceExclusionUnsupportedUnitArguments() {
        return Stream.of(
                Arguments.of(0, 0, 0f),      // 0 m
                Arguments.of(0, 1.521, 0f)  // ~ 105000 m
        );
    }

    private static Stream<Arguments> bandedDistanceKilometresArguments() {
        return Stream.of(
                Arguments.of(0, 0, 4f),     // 0 km - highest rating
                Arguments.of(0, 0.5, 3f),   // ~55 km
                Arguments.of(0, 1, 2f),     // ~111 km
                Arguments.of(0, 2, 1f),     // ~222 km
                Arguments.of(0, 3, 0f)      // ~333 km - lowest rating
        );
    }

    private static Stream<Arguments> bandedDistanceMilesArguments() {
        return Stream.of(
                Arguments.of(0, 0, 4f),     // 0 miles - highest rating
                Arguments.of(0, 0.3, 3f),   // ~20 miles
                Arguments.of(0, 0.8, 2f),   // ~55 miles
                Arguments.of(0, 1.5, 1f),   // ~103 miles
                Arguments.of(0, 2.2, 0f)    // ~152 miles - lowest rating
        );
    }

    private static Stream<Arguments> bandedDistanceUnsupportedUnitArguments() {
        return Stream.of(
                Arguments.of(0, 0, 0f),     // 0 m - unsupported unit returns 0
                Arguments.of(0, 1.521, 0f)  // ~105000 m - unsupported unit returns 0
        );
    }

    private static Stream<Arguments> locationTypeExclusionArguments() {
        return Stream.of(
                Arguments.of("Store", -1f),
                Arguments.of("Warehouse", -1f),
                Arguments.of("DifferentType", 1f)
        );
    }

    @ParameterizedTest
    @MethodSource("locationTypeExclusionArguments")
    void should_return_expected_rating_when_location_type_exclusion_criterion_is_applied(final String type, final float expected) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(Location.builder()
                                .type(type)
                                .build())
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationTypeExclusionCriterion();
        criterion.parseParams(mapper.valueToTree(ImmutableMap.of(VALUE, ImmutableList.of("Store", "Warehouse"))));
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> locationNetworkExclusionArguments() {
        return Stream.of(
                Arguments.of(ImmutableList.of("N1"), -1f),
                Arguments.of(ImmutableList.of("N2"), -1f),
                Arguments.of(ImmutableList.of("N3"), 1f),
                Arguments.of(ImmutableList.of("N1", "N3"), -1f)
        );
    }

    @ParameterizedTest
    @MethodSource("locationNetworkExclusionArguments")
    void should_return_expected_rating_when_location_network_exclusion_criterion_is_applied(final List<String> networks, final float expected) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(Location.builder()
                                .networks(networks)
                                .build())
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationNetworkExclusionCriterion();
        criterion.parseParams(mapper.valueToTree(ImmutableMap.of(VALUE, ImmutableList.of("N1", "N2"))));
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> dailyCapacityArguments() {
        return Stream.of(
                Arguments.of(10, 10f),
                Arguments.of(50, 50f),
                Arguments.of(100, 100f),
                Arguments.of(200, 200f)
        );
    }

    @ParameterizedTest
    @MethodSource("dailyCapacityArguments")
    void should_return_expected_rating_when_daily_capacity_criterion_is_applied(final Integer capacity, final float expected) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(Location.builder()
                                .attributes(Collections.singletonList(Attribute.builder()
                                        .name("DAILY_MAX_ORDER_CAPACITY")
                                        .type("INTEGER")
                                        .value(capacity)
                                        .build()))
                                .build())
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new LocationDailyCapacityCriterion();
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> networkPriorityArguments() {
        return Stream.of(
                Arguments.of(ImmutableList.of("N1"), 3f),
                Arguments.of(ImmutableList.of("N1", "N3"), 3f),
                Arguments.of(ImmutableList.of("N2"), 2f),
                Arguments.of(ImmutableList.of("N3"), 1f),
                Arguments.of(ImmutableList.of("N4"), 0f)
        );
    }

    @ParameterizedTest
    @MethodSource("networkPriorityArguments")
    void should_return_expected_rating_when_network_priority_criterion_is_applied(final List<String> networks, final float expected) {
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(Location.builder()
                                .networks(networks)
                                .build())
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new NetworkPriorityCriterion();
        criterion.parseParams(mapper.valueToTree(ImmutableMap.of(VALUE, ImmutableList.of("N1", "N2", "N3"))));
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> availabilityExclusionArguments() {
        return Stream.of(
                Arguments.of(100, 0, -1f),
                Arguments.of(100, 50, 1f),
                Arguments.of(100, 100, 1f),
                Arguments.of(100, 150, 1f)
        );
    }

    @ParameterizedTest
    @MethodSource("availabilityExclusionArguments")
    void should_return_expected_rating_when_availability_exclusion_criterion_is_applied(final Integer unfulfilledQuantity, final Integer lapQuantity, final float expected) {
        final int[] quantities = new int[1];
        quantities[0] = lapQuantity;
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .quantities(quantities)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .unfulfilledItems(ImmutableList.of(
                                OrderItem.builder()
                                        .quantity(unfulfilledQuantity)
                                        .build()
                        ))
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new InventoryAvailabilityExclusionCriterion();
        criterion.parseParams(mapper.valueToTree(ImmutableMap.of(VALUE, 50)));
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> availabilityArguments() {
        return Stream.of(
                Arguments.of(100, 0, 0f),
                Arguments.of(100, 50, 0.5f),
                Arguments.of(100, 100, 1f),
                Arguments.of(100, 150, 1.5f)
        );
    }

    @ParameterizedTest
    @MethodSource("availabilityArguments")
    void should_return_expected_rating_when_availability_criterion_is_applied(final Integer unfulfilledQuantity, final Integer lapQuantity, final float expected) {
        final int[] quantities = new int[1];
        quantities[0] = lapQuantity;
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .quantities(quantities)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .unfulfilledItems(ImmutableList.of(
                                OrderItem.builder()
                                        .quantity(unfulfilledQuantity)
                                        .build()
                        ))
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new InventoryAvailabilityCriterion();
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> bandedAvailabilityArguments() {
        return Stream.of(
                Arguments.of(100, 0, 0f),
                Arguments.of(100, 40, 1f),
                Arguments.of(100, 50, 1f),
                Arguments.of(100, 60, 2f),
                Arguments.of(100, 100, 2f),
                Arguments.of(100, 120, 3f),
                Arguments.of(100, 150, 3f),
                Arguments.of(100, 1000, 4f)
        );
    }

    @ParameterizedTest
    @MethodSource("bandedAvailabilityArguments")
    void should_return_expected_rating_when_banded_availability_criterion_is_applied(final Integer unfulfilledQuantity, final Integer lapQuantity, final float expected) {
        final int[] quantities = new int[1];
        quantities[0] = lapQuantity;
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .quantities(quantities)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .unfulfilledItems(ImmutableList.of(
                                OrderItem.builder()
                                        .quantity(unfulfilledQuantity)
                                        .build()
                        ))
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new InventoryAvailabilityBandedCriterion();
        criterion.parseParams(mapper.valueToTree(ImmutableMap.of(VALUE, ImmutableList.of(0, 50, 100, 150))));
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

    private static Stream<Arguments> orderValueArguments() {
        return Stream.of(
                Arguments.of(100, 0, 0.375f),
                Arguments.of(100, 50, 0.5f),
                Arguments.of(100, 100, 0.625f),
                Arguments.of(100, 1000, 0.625f)
        );
    }

    @ParameterizedTest
    @MethodSource("orderValueArguments")
    void should_return_expected_rating_when_order_value_criterion_is_applied(final Integer unfulfilledQuantity, final Integer lapQuantity, final float expected) {
        final int[] quantities = new int[2];
        quantities[0] = lapQuantity;
        quantities[1] = 50;
        SourcingCriteriaUtils.CriterionContext context = SourcingCriteriaUtils.DefaultCriterionContext.builder()
                .lap(SourcingUtils.LocationAndPositions.builder()
                        .location(LOCATION)
                        .quantities(quantities)
                        .build())
                .sourcingContext(DefaultSourcingContext.builder()
                        .unfulfilledItems(ImmutableList.of(
                                OrderItem.builder()
                                        .quantity(unfulfilledQuantity)
                                        .paidPrice(1.0)
                                        .build(),
                                OrderItem.builder()
                                        .quantity(100)
                                        .paidPrice(3.0)
                                        .build()
                        ))
                        .build())
                .build();

        final BaseSourcingCriterion criterion = new OrderValueCriterion();
        final float rating = criterion.apply(context);

        assertEquals(expected, rating, 0.001f);
    }

}
