/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fluentcommerce.graphql.sourcing.queries.fulfilmentoptions.GetFulfilmentOptionByIdQuery;
import com.fluentcommerce.graphql.sourcing.queries.location.GetFullLocationByRefQuery;
import com.fluentcommerce.graphql.sourcing.queries.order.GetOrderForSourcingQuery;
import com.fluentcommerce.graphql.sourcing.queries.product.GetProductsByRefQuery;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.SourcingUtils;
import com.fluentcommerce.util.test.executor.MockApiClient;
import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Answers;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.test.TestUtils.eventWithDefaults;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.Mockito.when;

@ExtendWith(MockitoExtension.class)
class SourcingContextUtilsTest {

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private Context context;

    @Mock
    private Entity entity;

    private static final ObjectMapper objectMapper = new ObjectMapper();

    static {
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
    }

    @BeforeEach
    public void beforeEach() {
        LocationUtils.resetCacheForTest();
        when(context.getEntity()).thenReturn(entity);
    }

    @Test
    void should_load_context_when_order_cc() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderCC.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults());

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextOrderCC.json", sourcingContext);
    }

    @Test
    void should_load_context_when_order_hd() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderHD.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults());

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextOrderHD.json", sourcingContext);
    }

    @Test
    void should_load_context_when_order_custom() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderCustom.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults());

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextOrderCustom.json", sourcingContext);
    }

    @Test
    void should_load_context_when_fulfilment_choice_hd() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderMulti.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults(Event.builder()
                .entityType("FULFILMENT_CHOICE")
                .entityId("2")
                .rootEntityId("1")
                .entityRef("FC_2")
                .build()));

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextFulfilmentChoiceHD.json", sourcingContext);
    }

    @Test
    void should_load_context_when_fulfilment_choice_custom() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderMultiWithCustomFulfilmentChoice.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults(Event.builder()
                .entityType("FULFILMENT_CHOICE")
                .entityId("2")
                .rootEntityId("1")
                .entityRef("FC_2")
                .build()));

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextFulfilmentChoiceCustom.json", sourcingContext);
    }

    @Test
    void should_load_context_when_fulfilment_options() throws IOException {
        final MockApiClient api = new MockApiClient()
                .mockNamedQuery(GetFulfilmentOptionByIdQuery.class, "graphql/responses/sourcing/SourcingFulfilmentOptions.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json")
                .mockNamedQuery(GetProductsByRefQuery.class, "graphql/responses/product/ProductsByRef.json");

        when(context.api()).thenReturn(api.get());
        when(context.getEvent()).thenReturn(eventWithDefaults(Event.builder()
                .entityType("FULFILMENT_OPTIONS")
                .entityId("2")
                .rootEntityId("1")
                .entityRef("FO_1")
                .build()));

        final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
        assertContextEquals("sourcing/SourcingContextFulfilmentOptions.json", sourcingContext);
    }

    private void assertContextEquals(String filename, final SourcingContext sourcingContext) throws IOException {
        String expectedContext;
        try (InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(filename);
             BufferedReader br = new BufferedReader(new InputStreamReader(is))) {
            expectedContext = br.lines().collect(Collectors.joining("\n"));
        }

        assertEquals(expectedContext, objectMapper.writeValueAsString(sourcingContext));
    }

}
