/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;


import com.fluentcommerce.graphql.sourcing.mutation.CreateFulfilmentMutation;
import com.fluentcommerce.graphql.sourcing.queries.inventory.GetVirtualPositionsQuery;
import com.fluentcommerce.graphql.sourcing.queries.location.GetFullLocationByRefQuery;
import com.fluentcommerce.graphql.sourcing.queries.location.GetFullLocationsByNetworkQuery;
import com.fluentcommerce.graphql.sourcing.queries.order.GetOrderForSourcingQuery;
import com.fluentcommerce.graphql.sourcing.queries.profile.GetProfileByRefQuery;
import com.fluentcommerce.util.RuleUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.SourcingContextUtils;
import com.fluentcommerce.util.sourcing.profile.SourcingProfile;
import com.fluentcommerce.util.test.executor.RuleContextGenerator;
import com.fluentcommerce.util.test.executor.RuleExecutor;
import com.fluentcommerce.util.test.executor.TestActions;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.rule.meta.ParamString;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.util.Collections;

import static com.fluentcommerce.model.Constants.EntityType.ENTITY_TYPE_ORDER;
import static com.fluentcommerce.model.Constants.PropertyField.PROP_SOURCING_PROFILE_REF;
import static com.fluentcommerce.model.Constants.PropertyField.PROP_SYSTEM_REJECTED_LOC_REF;
import static com.fluentcommerce.model.Constants.Status.ACTIVE;
import static com.fluentcommerce.util.sourcing.OrderUtils.createFulfilments;
import static com.fluentcommerce.util.sourcing.OrderUtils.fillFulfilmentType;
import static com.fluentcommerce.util.sourcing.SourcingUtils.findPlanBasedOnFallbackStrategies;
import static org.junit.jupiter.api.Assertions.assertEquals;

class SourcingUtilsRuleTest {

    private static final String FC_4 = "FC_4";
    private static final String CC_PFS = "CC_PFS";

    @BeforeEach
    public void resetCache() {
        LocationUtils.resetCacheForTest();
    }

    @Test
    void should_create_single_fulfilment_when_running_basic_sourcing_rule() {
        RuleContextGenerator context = RuleExecutor.of(CreateFulfilmentsBasedOnStrategies.class,
                        ImmutableMap.of(PROP_SOURCING_PROFILE_REF, "DEF"))
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderFromOneLocation.json")
                .mockNamedQuery(GetFullLocationsByNetworkQuery.class, "graphql/responses/location/GetLocationsByNetwork.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json")
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/GetVirtualPositions.json")
                .mockNamedQuery(GetProfileByRefQuery.class, "graphql/responses/profile/GetProfileByRef.json")
                .execute();

        assertEquals(0, context.getActionsOfType(TestActions.LogAction.class).size());
        assertEquals(1, context.getActionsOfType(TestActions.MutateAction.class).size());
        CreateFulfilmentMutation mutA = (CreateFulfilmentMutation) context.getActionsOfType(TestActions.MutateAction.class).get(0).getMutation();
        assertEquals(1, mutA.variables().input().items().size());
        assertEquals("5", mutA.variables().input().items().get(0).orderItem().id());
        assertEquals(2, mutA.variables().input().items().get(0).requestedQuantity());
        assertEquals(CC_PFS, mutA.variables().input().type());
        assertEquals(FC_4, mutA.variables().input().fulfilmentChoiceRef());
    }

    @Test
    void should_create_multiple_fulfilments_when_running_basic_sourcing_rule() {
        RuleContextGenerator context = RuleExecutor.of(CreateFulfilmentsBasedOnStrategies.class,
                        ImmutableMap.of(PROP_SOURCING_PROFILE_REF, "DEF"))
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderWithThreeItems.json")
                .mockNamedQuery(GetFullLocationsByNetworkQuery.class, "graphql/responses/location/GetLocationsByNetwork.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json")
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/GetVirtualPositions.json")
                .mockNamedQuery(GetProfileByRefQuery.class, "graphql/responses/profile/GetProfileByRef.json")
                .execute();

        assertEquals(0, context.getActionsOfType(TestActions.LogAction.class).size());

        assertEquals(2, context.getActionsOfType(TestActions.MutateAction.class).size());

        CreateFulfilmentMutation mutA = (CreateFulfilmentMutation) context.getActionsOfType(TestActions.MutateAction.class).get(0).getMutation();
        assertEquals(1, mutA.variables().input().items().size());
        assertEquals("5", mutA.variables().input().items().get(0).orderItem().id());
        assertEquals(2, mutA.variables().input().items().get(0).requestedQuantity());
        assertEquals(CC_PFS, mutA.variables().input().type());
        assertEquals(FC_4, mutA.variables().input().fulfilmentChoiceRef());

        CreateFulfilmentMutation mutB = (CreateFulfilmentMutation) context.getActionsOfType(TestActions.MutateAction.class).get(1).getMutation();
        assertEquals(1, mutB.variables().input().items().size());
        assertEquals("6", mutB.variables().input().items().get(0).orderItem().id());
        assertEquals(2, mutB.variables().input().items().get(0).requestedQuantity());
        assertEquals(CC_PFS, mutB.variables().input().type());
        assertEquals(FC_4, mutB.variables().input().fulfilmentChoiceRef());
    }

    @Test
    void should_create_single_fulfilment_when_running_fallback_sourcing_rule() {
        RuleContextGenerator context = RuleExecutor.of(CreateFulfilmentsBasedOnFallbackStrategies.class,
                        ImmutableMap.of(PROP_SOURCING_PROFILE_REF, "DEF"))
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderFromOneLocation.json")
                .mockNamedQuery(GetFullLocationsByNetworkQuery.class, "graphql/responses/location/GetLocationsByNetwork.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json")
                .mockNamedQuery(GetVirtualPositionsQuery.class, "graphql/responses/inventory/GetVirtualPositions_fallback.json")
                .mockNamedQuery(GetProfileByRefQuery.class, "graphql/responses/profile/GetProfileByRef.json")
                .execute();

        assertEquals(0, context.getActionsOfType(TestActions.LogAction.class).size());
        assertEquals(1, context.getActionsOfType(TestActions.MutateAction.class).size());
        CreateFulfilmentMutation mutA = (CreateFulfilmentMutation) context.getActionsOfType(TestActions.MutateAction.class).get(0).getMutation();
        assertEquals(1, mutA.variables().input().items().size());
        assertEquals("5", mutA.variables().input().items().get(0).orderItem().id());
        assertEquals(1, mutA.variables().input().items().get(0).requestedQuantity());
        assertEquals(CC_PFS, mutA.variables().input().type());
        assertEquals(FC_4, mutA.variables().input().fulfilmentChoiceRef());
    }

    @Test
    void should_not_create_rejected_fulfilment_when_no_unfulfilled_order_items() {
        RuleContextGenerator context = RuleExecutor.of(CreateRejectedFulfilment.class,
                        ImmutableMap.of(PROP_SYSTEM_REJECTED_LOC_REF, "F_NSYD"))
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderHD.json")
                .execute(Event.builder()
                        .entityType(ENTITY_TYPE_ORDER)
                        .entityId("242")
                        .build());
        assertEquals(0, context.getActionsOfType(TestActions.MutateAction.class).size());
    }

    @Test
    void should_create_rejected_fulfilment_when_unfulfilled_order_items_exist() {
        RuleContextGenerator context = RuleExecutor.of(CreateRejectedFulfilment.class,
                        ImmutableMap.of(PROP_SYSTEM_REJECTED_LOC_REF, "F_NSYD"))
                .mockNamedQuery(GetOrderForSourcingQuery.class, "graphql/responses/sourcing/SourcingOrderHDUnfulfilledOrderItems.json")
                .mockNamedQuery(GetFullLocationByRefQuery.class, "graphql/responses/location/Location_F_NSYD.json")
                .execute(Event.builder()
                        .entityType(ENTITY_TYPE_ORDER)
                        .entityId("242")
                        .build());
        assertEquals(1, context.getActionsOfType(TestActions.MutateAction.class).size());
        CreateFulfilmentMutation mutA = (CreateFulfilmentMutation) context.getActionsOfType(TestActions.MutateAction.class).get(0).getMutation();
        assertEquals(1, mutA.variables().input().items().size());
        assertEquals("33669", mutA.variables().input().fromAddress().id());
        assertEquals("HD_PFS", mutA.variables().input().type());
    }

    @RuleInfo(name = "CreateFulfilmentsBasedOnStrategies", description = "Fulfil all based on profile {profileRef}")
    @ParamString(name = PROP_SOURCING_PROFILE_REF, description = "Ref of the profile to source from")
    public static class CreateFulfilmentsBasedOnStrategies implements Rule {
        @Override
        public void run(Context context) {
            SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);
            if (sourcingProfile.getSourcingStrategies() == null) {
                return;
            }

            SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
            SourcingUtils.SourcingPlan plan = SourcingUtils.findPlanBasedOnStrategies(context, sourcingContext, sourcingProfile, ImmutableList.of(ACTIVE), null);
            if (plan.getFulfilments() != null) {
                fillFulfilmentType(sourcingContext, plan.getFulfilments());
                createFulfilments(context, sourcingContext, plan.getFulfilments());
            }
        }
    }

    @RuleInfo(name = "CreateFulfilmentsBasedOnFallbackStrategies", description = "Fulfil as many as possible based on profile {profileRef}")
    @ParamString(name = PROP_SOURCING_PROFILE_REF, description = "Ref of the profile to source from")
    public static class CreateFulfilmentsBasedOnFallbackStrategies implements Rule {
        @Override
        public void run(Context context) {
            SourcingProfile sourcingProfile = SourcingUtils.loadSourcingProfile(context);
            if (sourcingProfile.getSourcingFallbackStrategies() == null) {
                return;
            }

            SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
            SourcingUtils.SourcingPlan plan = findPlanBasedOnFallbackStrategies(context, sourcingContext, sourcingProfile,
                    ImmutableList.of(ACTIVE), null);
            if (plan.getFulfilments() != null) {
                fillFulfilmentType(sourcingContext, plan.getFulfilments());
                createFulfilments(context, sourcingContext, plan.getFulfilments());
            }
        }
    }

    @RuleInfo(
            name = "CreateRejectedFulfilment",
            description = "This rule creates a rejected Fulfilment for an item of a sourcing request that cannot be fulfilled.")
    @ParamString(name = PROP_SYSTEM_REJECTED_LOC_REF,
            description = "Reference of the Location where the rejected Fulfilment will be created.")
    public static class CreateRejectedFulfilment implements Rule {
        @Override
        public void run(Context context) {
            RuleUtils.validateRulePropsIsNotEmpty(context, PROP_SYSTEM_REJECTED_LOC_REF);
            final SourcingContext sourcingContext = SourcingContextUtils.loadSourcingContext(context, SourcingUtils::getUnfulfilledItems);
            SourcingUtils.Fulfilment rejectedFulfilment = SourcingUtils.buildRejectedFulfilment(context, sourcingContext, context.getProp(PROP_SYSTEM_REJECTED_LOC_REF));
            if (rejectedFulfilment != null) {
                fillFulfilmentType(sourcingContext, Collections.singletonList(rejectedFulfilment));
                createFulfilments(context, sourcingContext, Collections.singletonList(rejectedFulfilment));
            }
        }
    }
}
