/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;

/**
 * Represents a contract for evaluating a sourcing criterion against a given context.
 *
 * <p>Implementations of this interface define specific business rules that are
 * evaluated dynamically during the sourcing process.</p>
 */
public interface SourcingCriterion {

    float RATING_EXCLUDE = -1f;

    /**
     * Applies the criterion function to a criterion context.
     *
     * @param criterionContext context of criterion execution, including Sourcing Context and location information
     * @return "rating" of the location in the context
     */
    float apply(final SourcingCriteriaUtils.CriterionContext criterionContext);

    /**
     * Parses and saves configuration params of the Sourcing Criterion.
     *
     * @param params JsonNode with the configuration params
     */
    default void parseParams(final JsonNode params) {
    }

    /**
     * Normalizes the rating value to the range [0.0f, 1.0f] according to the Sourcing Criterion logic (i.e. closer distance is better).
     * Default implementation assumes higher rating is better.
     *
     * @param min    minimum rating among the values that are being normalized
     * @param max    maximum rating among the values that are being normalized
     * @param rating rating of the location that should be normalized
     * @return rating of the location normalized to range [0.0f, 1.0f]
     */
    default float normalize(final float min, final float max, final float rating) {

        if (rating == RATING_EXCLUDE) {
            return RATING_EXCLUDE;
        }
        if (Float.compare(min, max) == 0) {
            return Float.compare(min, 0f) > 0 ? 1f : 0f;
        }
        return (rating - min) / (max - min);
    }

    /**
     * Builds a JSON representation of the setting based on {@link SourcingCriterionInfo} annotations
     *
     * @return JSON representation of the setting
     */
    default ObjectNode toJson() {
        return null;
    }

}
