/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.SourcingUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.profile.SourcingStrategy;
import com.fluentcommerce.util.units.DistanceMeasurementUnits;
import com.google.common.collect.ImmutableList;
import lombok.AccessLevel;
import lombok.Builder;
import lombok.NoArgsConstructor;
import lombok.Value;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import static java.util.stream.Collectors.toList;
import static com.fluentcommerce.util.units.DistanceMeasurementUnits.KILOMETRES;

/**
 * Utility class that provides helper methods for evaluating sourcing criteria.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing strategy execution.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class SourcingCriteriaUtils {

    /**
     * Container for a list of stacked Sourcing Criteria.
     * <p>
     * Criteria can be stacked to consider multiple criteria when sourcing.
     * They are applied and evaluated in order, so the first criterion is applied, then the second used as a tie-breaker
     * for the first, and so on.
     * <p>
     * For example: in the scenario where we have two Criteria:
     * <ol>
     * <li>the first returns 1.0f for stores with a given attribute and 0.5f for others</li>
     * <li>the second returns the distance from the delivery address in kms</li>
     * </ol>
     * Then all locations with the attribute will rank higher than all stores without, but within those groups the
     * ranking will be determined by distance from the delivery address.
     * <p>
     * If instead you want to weigh multiple factors at the same time, create a new `Criterion` that combines
     * those values into a single float according to your requirements.
     */
    @Value
    public static class CriteriaArray {
        List<SourcingCriterion> criteria;

        public static CriteriaArray of(final SourcingCriterion... criteria) {
            return new CriteriaArray(ImmutableList.copyOf(criteria));
        }

        public static CriteriaArray of(final List<SourcingCriterion> criteria) {
            return new CriteriaArray(ImmutableList.copyOf(criteria));
        }

        /**
         * Applies an array of criteria to a Criterion Context
         *
         * @param criterionContext Criterion Context containing SourcingContext and Location data
         * @return array of ratings for each criterion applied to the Criterion Context
         */
        public float[] apply(final CriterionContext criterionContext) {
            float[] rating = new float[criteria.size()];
            for (int i = 0; i < criteria.size(); i++) {
                rating[i] = criteria.get(i).apply(criterionContext);
            }
            return rating;
        }

        /**
         * Normalizes the rating values for the same criterion to the [0f, 1f] range.
         * The result rating is calculated as (rating - MIN) / (MAX - MIN) where MIN and MAX are
         * minimum and maximum rating for the specific criterion
         *
         * @param ratingsList list of ratings for locations, each item on the list contains ratings for all criteria in a single location
         */
        public void normalize(final List<float[]> ratingsList) {
            final int size = criteria.size();
            final float[] min = new float[size];
            final float[] max = new float[size];

            Arrays.fill(min, Float.POSITIVE_INFINITY);
            Arrays.fill(max, Float.NEGATIVE_INFINITY);

            // Calculate min and max for each criterion, skipping RATING_EXCLUDE
            for (float[] ratings : ratingsList) {
                for (int i = 0; i < size; i++) {
                    float rating = ratings[i];
                    if (rating != SourcingCriterion.RATING_EXCLUDE) {
                        min[i] = Math.min(rating, min[i]);
                        max[i] = Math.max(rating, max[i]);
                    }
                }
            }

            // Normalize using criterion-specific logic
            for (float[] ratings : ratingsList) {
                for (int i = 0; i < size; i++) {
                    ratings[i] = criteria.get(i).normalize(min[i], max[i], ratings[i]);
                }
            }
        }
    }

    /**
     * Default implementation of the {@link CriterionContext}
     */
    @Value
    @Builder
    public static class DefaultCriterionContext implements CriterionContext {
        SourcingUtils.LocationAndPositions lap;
        SourcingContext sourcingContext;
    }

    /**
     * An interface representing the information which is passed to the Sourcing Criteria upon evaluation
     */
    public interface CriterionContext {
        SourcingContext getSourcingContext();

        SourcingUtils.LocationAndPositions getLap();
    }

    /**
     * Extracts an array of float values from criterion params, usually representing "banded" values
     *
     * @param params    JsonNode containing criterion params (configuration)
     * @param fieldName name of the param field to extract
     * @return extracted array of float values from params
     */
    @NotNull
    public static Float[] getFloats(final JsonNode params, final String fieldName) {
        JsonNode array = params.get(fieldName);

        return array != null && array.isArray() ? StreamSupport.stream(array.spliterator(), false)
                .filter(JsonNode::isNumber)
                .map(JsonNode::asDouble)
                .map(Double::floatValue)
                .toArray(Float[]::new) : new Float[]{};
    }

    /**
     * Extracts and parses distance measurement unit from criterion parameters.
     * <p>
     * Searches for the specified field in the JSON parameters and converts it to a
     * {@link DistanceMeasurementUnits} enum value. The method is case-insensitive
     * and handles missing or invalid values gracefully.
     * <p>
     * <strong>Behavior:</strong>
     * <ul>
     *   <li>If {@code params} is null: returns KILOMETRES</li>
     *   <li>If field is missing: returns KILOMETRES</li>
     *   <li>If field is not text: returns KILOMETRES</li>
     *   <li>If unit is unknown: returns KILOMETRES</li>
     * </ul>
     *
     * @param params    JsonNode containing criterion params (configuration)
     * @param fieldName name of the param field to extract
     * @return distance measurement unit, defaults to KILOMETRES if parsing fails
     * @see DistanceMeasurementUnits#fromString(String)
     */
    public static DistanceMeasurementUnits getDistanceUnits(final JsonNode params, final String fieldName) {

        return DistanceMeasurementUnits.fromString(Optional.ofNullable(params)
                .map(p -> p.get(fieldName))
                .filter(JsonNode::isTextual)
                .map(JsonNode::asText).orElse(KILOMETRES.name()));
    }

    /**
     * Extracts an array of String values from criterion params, usually representing "banded" values
     *
     * @param params    JsonNode containing criterion params (configuration)
     * @param fieldName name of the param field to extract
     * @return extracted array of String values from params
     */
    @NotNull
    public static List<String> getStrings(final JsonNode params, final String fieldName) {
        JsonNode bandsArray = params.get(fieldName);

        return bandsArray != null && bandsArray.isArray() ? StreamSupport.stream(bandsArray.spliterator(), false)
                .filter(JsonNode::isTextual)
                .map(JsonNode::textValue)
                .collect(Collectors.toList()) : Collections.emptyList();
    }

    /**
     * Builds a {@link CriteriaArray} of Sourcing Criteria based on configurations in the provided Sourcing Strategy and Default Criteria
     *
     * @param strategy        Sourcing Strategy with configurations
     * @param defaultCriteria Array of Default Sourcing Criteria
     * @return a {@link CriteriaArray} of Sourcing Criteria based on provided configurations
     */
    public static CriteriaArray getCriteria(final SourcingStrategy strategy, final CriteriaArray defaultCriteria) {
        if (strategy == null || strategy.getSourcingCriteria() == null) {
            return defaultCriteria;
        }
        CriteriaArray strategyCriteria = CriteriaArray.of(strategy.getSourcingCriteria()
                .stream()
                .map(c -> {
                    final JsonNode node = Optional.ofNullable(c.getParams())
                            .map(JsonUtils::objectToNode)
                            .orElseGet(JsonNodeFactory.instance::objectNode);
                    final SourcingCriterion criterion = SourcingCriteriaTypeRegistry.getSourcingCriterion(c.getType());
                    criterion.parseParams(node);
                    return criterion;
                })
                .collect(toList()));

        List<SourcingCriterion> mergedCriteria = new ArrayList<>();
        mergedCriteria.addAll(defaultCriteria.getCriteria() != null ? defaultCriteria.getCriteria() : Collections.emptyList());
        mergedCriteria.addAll(strategyCriteria.getCriteria() != null ? strategyCriteria.getCriteria() : Collections.emptyList());
        return CriteriaArray.of(mergedCriteria);
    }

    /**
     * Builds a {@link CriteriaArray} of Default Sourcing Criteria based on the provided criterions
     *
     * @return a {@link CriteriaArray} of Default Sourcing Criteria
     */
    public static CriteriaArray getDefaultCriteria() {
        final RejectedLocationExclusionCriterion locationExclusionCriterion = new RejectedLocationExclusionCriterion();
        return CriteriaArray.of(locationExclusionCriterion);
    }

}
