/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fluentcommerce.util.sourcing.SourcingCriteriaSettingGenerator;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.SneakyThrows;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Registry for mapping Sourcing Criteria type names to their corresponding implementation classes.
 *
 * <p>This utility class provides methods to register and retrieve sourcing criteria types
 * by their string identifiers. It is used during sourcing strategy evaluation to dynamically
 * instantiate criteria implementations based on type names.</p>
 *
 * <p>The registry is thread-safe and uses a {@link ConcurrentHashMap} to store mappings.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class SourcingCriteriaTypeRegistry {

    @Getter
    private static final Map<String, Class<? extends SourcingCriterion>> typeNameToClassMap = new ConcurrentHashMap<>();

    /**
     * Registers the Sourcing Criterion class for the provided typeName
     *
     * @param typeName name of the type to save the Criterion for
     * @param clazz    class which implements the Criterion
     */
    public static void register(final String typeName, final Class<? extends SourcingCriterion> clazz) {
        typeNameToClassMap.put(typeName, clazz);
    }

    /**
     * Retrieve the Criterion class corresponding to the provided typeName
     *
     * @param typeName type of the Criterion class to retrieve
     * @return the Criterion class corresponding to the provided typeName
     */
    public static Class<? extends SourcingCriterion> getSourcingCriterionType(final String typeName) {

        Class<? extends SourcingCriterion> clazz = typeNameToClassMap.get(typeName);
        if (clazz == null) {
            throw new IllegalArgumentException("Unknown Sourcing Criterion type: " + typeName);
        }

        return clazz;
    }

    /**
     * Retrieves an instance of the Criterion class corresponding to the provided typeName
     *
     * @param typeName type of the Criterion class to retrieve instance of
     * @return an instance of the Criterion class corresponding to the provided typeName
     */
    @SneakyThrows
    public static SourcingCriterion getSourcingCriterion(final String typeName) {
        final Class<?> clazz = getSourcingCriterionType(typeName);
        return (SourcingCriterion) clazz.getConstructor().newInstance();
    }

    static {
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationDistanceExclusion",
                LocationDistanceExclusionCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationTypeExclusion",
                LocationTypeExclusionCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationDistance",
                LocationDistanceCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationDistanceBanded",
                LocationDistanceBandedCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationDailyCapacity",
                LocationDailyCapacityCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.networkPriority",
                NetworkPriorityCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.inventoryAvailabilityExclusion",
                InventoryAvailabilityExclusionCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.inventoryAvailability",
                InventoryAvailabilityCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.inventoryAvailabilityBanded",
                InventoryAvailabilityBandedCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.orderValue",
                OrderValueCriterion.class);
        SourcingCriteriaTypeRegistry.register("fc.sourcing.criterion.locationNetworkExclusion",
                LocationNetworkExclusionCriterion.class);
    }

    /**
     * Executes the sourcing criterion JSON generator.
     * <p>
     * This method scans all registered sourcing criterion types, aggregates the resulting JSON objects to pretty-printed JSON SourcingCriteriaSetting.json
     * <p>
     *
     * @throws IOException if an error occurs during criterion processing or file writing
     * @throws URISyntaxException if an error occurs during criterion processing or file writing
     */
    public static void generateSourcingCriteriaSetting() throws IOException, URISyntaxException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.enable(SerializationFeature.INDENT_OUTPUT);

        ArrayNode resultArray = mapper.createArrayNode();
        for (String typeName : SourcingCriteriaTypeRegistry.getTypeNameToClassMap().keySet()) {
            SourcingCriterion criterion = SourcingCriteriaTypeRegistry.getSourcingCriterion(typeName);
            resultArray.add(criterion.toJson());
        }

        Path jarOrClasses = Paths.get(
                SourcingCriteriaSettingGenerator.class.getProtectionDomain().getCodeSource().getLocation().toURI()
        );
        Path baseDir = Files.isRegularFile(jarOrClasses) ? jarOrClasses.getParent() : jarOrClasses;

        File outputFile = baseDir.resolve("SourcingCriteriaSetting.json").toFile();
        outputFile.getParentFile().mkdirs();
        mapper.writeValue(outputFile, resultArray);
    }
}