/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.criterion;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.SourcingUtils;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionInfo;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParam;
import com.fluentcommerce.util.sourcing.criterion.annotation.SourcingCriterionParamSelectComponentOption;
import com.fluentcommerce.util.units.DistanceMeasurementUnits;

import java.util.Optional;

import static com.fluentcommerce.util.sourcing.criterion.SourcingCriteriaUtils.getDistanceUnits;
import static com.fluentcommerce.util.units.DistanceMeasurementUnits.KILOMETRES;
import static com.fluentcommerce.util.units.DistanceMeasurementUnits.MILES;

@SourcingCriterionInfo(name = "locationDistanceExclusion",
        type = "fc.sourcing.criterion.locationDistanceExclusion",
        tags = {"Exclusion", "ATS-agnostic"})
public class LocationDistanceExclusionCriterion extends BaseSourcingCriterion {

    @SourcingCriterionParam(name = "value", component = "input")
    private double maxDistance;
    @SourcingCriterionParam(
            name = "valueUnit", 
            component = "select",
            selectComponentOptions = {
                    @SourcingCriterionParamSelectComponentOption(
                            label = "fc.sourcing.criterion.locationDistanceExclusion.valueunit.kilometres",
                            value = "kilometres"
                    ),
                    @SourcingCriterionParamSelectComponentOption(
                            label = "fc.sourcing.criterion.locationDistanceExclusion.valueunit.miles",
                            value = "miles"
                    )
            }
    )
    private DistanceMeasurementUnits distanceUnit;

    @Override
    public void parseParams(final JsonNode params) {

        maxDistance = Optional.ofNullable(params)
                .map(p -> p.get("value"))
                .filter(JsonNode::isNumber)
                .map(JsonNode::asDouble)
                .orElse(Double.MAX_VALUE);

        distanceUnit = getDistanceUnits(params, "valueUnit");
    }

    @Override
    protected float execute(final SourcingCriteriaUtils.CriterionContext criterionContext) {

        if(distanceUnit != KILOMETRES && distanceUnit != MILES) {
            return 0;
        }

        final Double lat = Optional.ofNullable(criterionContext.getSourcingContext())
                .map(SourcingContext::getFulfilmentChoice)
                .map(FulfilmentChoice::getAddress)
                .map(Address::getLatitude)
                .orElse(null);
        final Double lon = Optional.ofNullable(criterionContext.getSourcingContext())
                .map(SourcingContext::getFulfilmentChoice)
                .map(FulfilmentChoice::getAddress)
                .map(Address::getLongitude)
                .orElse(null);

        if (lat == null || lon == null) {
            return 0;
        }

        final Double locationLat = Optional.ofNullable(criterionContext.getLap())
                .map(SourcingUtils.LocationAndPositions::getLocation)
                .map(Location::getPrimaryAddress)
                .map(Address::getLatitude)
                .orElse(null);
        final Double locationLon = Optional.ofNullable(criterionContext.getLap())
                .map(SourcingUtils.LocationAndPositions::getLocation)
                .map(Location::getPrimaryAddress)
                .map(Address::getLongitude)
                .orElse(null);

        if (locationLat == null || locationLon == null) {
            return 0;
        }

        final double distance = LocationUtils.calculateDistance(lat, lon, locationLat, locationLon, distanceUnit);

        return distance >= maxDistance ? -1f : 1f;
    }
}
