/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context.loader;

import com.fluentcommerce.graphql.sourcing.queries.fulfilmentoptions.GetFulfilmentOptionByIdQuery;
import com.fluentcommerce.graphql.sourcing.queries.product.GetProductsByRefQuery;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.LocationUtils;
import com.fluentcommerce.util.sourcing.context.DefaultSourcingContext;
import com.fluentcommerce.util.sourcing.context.SourcingContext;
import com.fluentcommerce.util.sourcing.context.UnfulfilledItemProcessor;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.FulfilmentChoice;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OrderItem;
import com.fluentcommerce.util.sourcing.context.model.Product;
import com.fluentretail.api.model.sku.Price;
import com.fluentretail.rubix.v2.context.Context;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.sourcing.context.SourcingContextUtils.mapAttributes;

/**
 * {@link SourcingContextLoader} used for loading context for "FULFILMENT_OPTIONS" entity type
 */
public class FulfilmentOptionsLoader extends AbstractSourcingContextLoader<GetFulfilmentOptionByIdQuery.Data> {
    @Override
    protected GetFulfilmentOptionByIdQuery.Data loadResponse(final Context context) {
        // CC or HD Fulfilment Options workflow
        return (GetFulfilmentOptionByIdQuery.Data)
                context.api().query(GetFulfilmentOptionByIdQuery.builder()
                        .id(context.getEvent().getEntityId())
                        .build());
    }

    @Override
    protected SourcingContext mapResponse(final GetFulfilmentOptionByIdQuery.Data fulfilmentOptionResponse,
                                          final Context context,
                                          final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        final SourcingContext sourcingContext = map(fulfilmentOptionResponse.fulfilmentOption(), context);
        return Optional.ofNullable(sourcingContext)
                .map(SourcingContext::toBuilder)
                .map(builder -> builder
                        .unfulfilledItems(unfulfilledItemProcessor.process(sourcingContext))
                        .build())
                .orElse(null);
    }

    private SourcingContext map(final GetFulfilmentOptionByIdQuery.FulfilmentOption fulfilmentOption,
                                final Context context) {
        if (fulfilmentOption == null) {
            return DefaultSourcingContext.builder().build();
        }

        final Location pickupLocation = LocationUtils.getLocationByRef(
                context, Optional.of(fulfilmentOption)
                        .map(GetFulfilmentOptionByIdQuery.FulfilmentOption::locationRef)
                        .orElse(null));

        final Map<Pair<String, String>, Integer> productQuantities = Optional.ofNullable(fulfilmentOption.products())
                .map(p -> p.stream()
                        .collect(Collectors.toMap(
                                product -> Pair.of(product.productRef(), product.catalogueRef()),
                                GetFulfilmentOptionByIdQuery.Product::requestedQuantity)))
                .orElse(Collections.emptyMap());
        final List<Product> products =
                Optional.ofNullable(fulfilmentOption.products())
                        .map(p -> p.stream()
                                .map(GetFulfilmentOptionByIdQuery.Product::productRef)
                                .collect(Collectors.toList()))
                        .map(productRefs -> (GetProductsByRefQuery.Data) context.api().query(GetProductsByRefQuery.builder()
                                .productRefs(productRefs)
                                .build()))
                        .map(GetProductsByRefQuery.Data::products)
                        .map(this::mapProducts)
                        .orElse(null);

        return DefaultSourcingContext.builder()
                .createdOn((Date) fulfilmentOption.createdOn())
                .updatedOn((Date) fulfilmentOption.updatedOn())
                .attributes(mapAttributes(fulfilmentOption.attributes()))
                .type(fulfilmentOption.orderType())
                .fulfilmentChoice(FulfilmentChoice.builder()
                        .pickupLocationRef(fulfilmentOption.locationRef())
                        .pickupLocation(pickupLocation)
                        .address(ObjectUtils.firstNonNull(
                                Optional.ofNullable(pickupLocation)
                                        .map(Location::getPrimaryAddress)
                                        .orElse(null),
                                // Address is mapped differently due to field name mismatch (street vs addressLine)
                                Optional.ofNullable(fulfilmentOption.address())
                                        .map(address -> Address.builder()
                                                .city(address.city())
                                                .companyName(address.companyName())
                                                .country(address.country())
                                                .name(address.name())
                                                .state(address.state())
                                                .postcode(address.postcode())
                                                .longitude(address.longitude())
                                                .latitude(address.latitude())
                                                .street(address.addressLine1())
                                                .street2(address.addressLine2())
                                                .build())
                                        .orElse(null)
                        ))
                        .build())
                .items(Optional.ofNullable(products)
                        .map(p -> p.stream()
                                .filter(product -> productQuantities.containsKey(Pair.of(product.getRef(), product.getCatalogueRef())))
                                .map(product -> OrderItem.builder()
                                        .product(product)
                                        .quantity(productQuantities.get(Pair.of(product.getRef(), product.getCatalogueRef())))
                                        .build())
                                .collect(Collectors.toList()))
                        .orElse(null)
                )
                .build();
    }

    private List<Product> mapProducts(final GetProductsByRefQuery.Products products) {
        if (products == null || products.edges() == null || products.edges().isEmpty()) {
            return null;
        }
        return products.edges().stream()
                .map(GetProductsByRefQuery.Edge::node)
                .map(product -> Product.builder()
                        .id(product.id())
                        .createdOn((Date) product.createdOn())
                        .updatedOn((Date) product.updatedOn())
                        .ref(product.ref())
                        .type(product.type())
                        .status(product.status())
                        .name(product.name())
                        .summary(product.summary())
                        .attributes(mapAttributes(product.attributes()))
                        .prices(Optional.ofNullable(product.prices())
                                .map(prices -> prices.stream()
                                        .map(price -> JsonUtils.anyToPojo(price, Price.class))
                                        .collect(Collectors.toList()))
                                .orElse(null)
                        )
                        .tax(Optional.ofNullable(product.tax())
                                .map(tax -> JsonUtils.anyToPojo(tax, Product.Tax.class))
                                .orElse(null))
                        .categories(mapCategories(product.categories()))
                        .catalogueRef(product.catalogue().ref())
                        .build())
                .collect(Collectors.toList());
    }

    private List<Product.Category> mapCategories(GetProductsByRefQuery.Categories categories) {
        if (categories == null || categories.categoryEdges() == null || categories.categoryEdges().isEmpty()) {
            return null;
        }
        return categories.categoryEdges().stream()
                .map(GetProductsByRefQuery.CategoryEdge::categoryNode)
                .map(categoryNode -> JsonUtils.anyToPojo(categoryNode, Product.Category.class))
                .collect(Collectors.toList());
    }
}
