/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.context;

import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.context.loader.FulfilmentChoiceLoader;
import com.fluentcommerce.util.sourcing.context.loader.FulfilmentOptionsLoader;
import com.fluentcommerce.util.sourcing.context.loader.OrderContextLoader;
import com.fluentcommerce.util.sourcing.context.loader.SourcingContextLoader;
import com.fluentretail.api.model.attribute.Attribute;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import static com.fluentcommerce.model.Constants.EntityType.ENTITY_TYPE_FULFILMENT_CHOICE;
import static com.fluentcommerce.model.Constants.EntityType.ENTITY_TYPE_FULFILMENT_OPTIONS;
import static com.fluentcommerce.model.Constants.EntityType.ENTITY_TYPE_ORDER;

/**
 * Utility class that provides helper methods for working with {@link SourcingContext}.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing rule execution.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class SourcingContextUtils {

    private static final Map<String, SourcingContextLoader> DEFAULT_LOADERS = new HashMap<>();

    static {
        DEFAULT_LOADERS.put(ENTITY_TYPE_ORDER.toLowerCase(Locale.getDefault()), new OrderContextLoader());
        DEFAULT_LOADERS.put(ENTITY_TYPE_FULFILMENT_CHOICE.toLowerCase(Locale.getDefault()), new FulfilmentChoiceLoader());
        DEFAULT_LOADERS.put(ENTITY_TYPE_FULFILMENT_OPTIONS.toLowerCase(Locale.getDefault()), new FulfilmentOptionsLoader());
    }

    private static final Map<String, SourcingContextLoader> loaderRegistry = new ConcurrentHashMap<>(DEFAULT_LOADERS);

    /**
     * Registers a custom context loader for the specified entity type.
     *
     * @param entityType the event entity type (e.g. "ORDER", "FULFILMENT_CHOICE", etc.).
     *                   The key is normalized to lower case.
     * @param loader     the custom ContextLoader to use
     */
    public static void registerContextLoader(final String entityType, final SourcingContextLoader loader) {
        loaderRegistry.put(entityType.toLowerCase(Locale.getDefault()), loader);
    }

    /**
     * Loads SourcingContext of the sourcing request (i.e. Order, Fulfilment Choice, etc.) based on the Event Context
     * and applies the provided unfulfilledItemProcessor to it
     *
     * @param context                  Event context to load from
     * @param unfulfilledItemProcessor Logic for calculating unfulfilled items based on SourcingContext
     * @return Loaded SourcingContext with calculated unfulfilled items
     */
    public static SourcingContext loadSourcingContext(final Context context,
                                                      final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        final Event event = context.getEvent();
        final String key = event.getEntityType().toLowerCase(Locale.getDefault());
        final SourcingContextLoader loader = loaderRegistry.get(key);
        if (loader != null) {
            return loader.load(context, unfulfilledItemProcessor);
        }
        return null;
    }

    /**
     * Creates a new SourcingContext copying the original context and supplying a different
     * unfulfilled items calculation logic.
     *
     * @param other                    The original SourcingContext
     * @param unfulfilledItemProcessor Logic for calculating unfulfilled items based on SourcingContext
     * @return Copied SourcingContext with new unfulfilled items
     */
    public static SourcingContext from(final SourcingContext other,
                                       final UnfulfilledItemProcessor unfulfilledItemProcessor) {
        return other.toBuilder()
                .unfulfilledItems(unfulfilledItemProcessor.process(other))
                .build();
    }

    /**
     * Maps an Apollo GraphQL response object to a list of {@link Attribute} objects
     *
     * @param rawAttributes raw Apollo GraphQL response object for attributes
     * @return list of {@link Attribute} objects
     */
    public static List<Attribute> mapAttributes(final Object rawAttributes) {
        return Optional.ofNullable(rawAttributes)
                .map(obj -> {
                    @SuppressWarnings("unchecked") final Iterable<Object> iterable = (Iterable<Object>) obj;
                    return StreamSupport.stream(iterable.spliterator(), false)
                            .map(SourcingContextUtils::mapToAttribute)
                            .collect(Collectors.toList());
                })
                .orElse(null);
    }

    private static Attribute mapToAttribute(final Object source) {
        return JsonUtils.anyToPojo(source, Attribute.class);
    }
}
