/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;


import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Registry responsible for mapping operator names (e.g., "in", "greater_than")
 * to their corresponding {@link SourcingConditionOperator} implementations.
 *
 * <p>This class manages both built-in and custom sourcing condition operators used
 * during sourcing condition evaluations.</p>
 *
 * <p>It provides:</p>
 * <ul>
 *   <li>Built-in registration of standard operators (e.g., equality, range, membership checks)</li>
 *   <li>Support for dynamic registration of custom operators at runtime</li>
 *   <li>Thread-safe access via an internal {@link ConcurrentHashMap}</li>
 * </ul>
 *
 * <p>This allows flexible extension of sourcing logic through domain-specific operator logic.</p>
 *
 * <p>Usage example:</p>
 * <pre>{@code
 * SourcingConditionOperatorRegistry.register("custom_operator", context -> {
 *     // implement condition logic
 *     return true;
 * });
 * }</pre>
 *
 * <p>To retrieve an operator by name:</p>
 * <pre>{@code
 * SourcingConditionOperator op = SourcingConditionOperatorRegistry.getSourcingConditionOperator("in");
 * }</pre>
 */
public final class SourcingConditionOperatorRegistry {
    private static final Map<String, SourcingConditionOperator> typeToOperatorMap = new ConcurrentHashMap<>();

    private SourcingConditionOperatorRegistry() {
    }

    static {
        typeToOperatorMap.put("equals", SourcingConditionMathOperator::ifValueEquals);
        typeToOperatorMap.put("not_equals", SourcingConditionMathOperator::ifValueNotEquals);
        typeToOperatorMap.put("greater_than", SourcingConditionMathOperator::ifValueIsGreaterThan);
        typeToOperatorMap.put("greater_than_or_equals", SourcingConditionMathOperator::ifValueIsGreaterThanOrEquals);
        typeToOperatorMap.put("less_than", SourcingConditionMathOperator::ifValueIsLessThan);
        typeToOperatorMap.put("less_than_or_equals", SourcingConditionMathOperator::ifValueIsLessThanOrEquals);
        typeToOperatorMap.put("in", SourcingConditionMembershipOperator::ifValueIn);
        typeToOperatorMap.put("not_in", SourcingConditionMembershipOperator::ifValueNotIn);
        typeToOperatorMap.put("between", SourcingConditionBetweenOperator::ifValueBetween);
        typeToOperatorMap.put("exists", SourcingConditionExistsOperator::ifExists);
    }

    /**
     * Retrieves the {@link SourcingConditionOperator} implementation associated with the given operator name.
     * <p>
     * This method is typically used during condition evaluation to resolve the correct logic handler
     * for standard or custom operators (e.g., "in", "greater_than", "custom_operator").
     *
     * @param operatorName the name of the operator to retrieve; must not be {@code null}
     * @return the corresponding {@link SourcingConditionOperator}
     * @throws IllegalArgumentException if no operator is registered for the given name
     */
    public static SourcingConditionOperator getSourcingConditionOperator(final String operatorName) {

        Objects.requireNonNull(operatorName, "Operator name must not be null");

        SourcingConditionOperator operator = typeToOperatorMap.get(operatorName);

        if (operator == null) {
            throw new IllegalArgumentException(String.format("Unknown operator name: %s", operatorName));
        }

        return operator;
    }

    /**
     * Registers a new {@link SourcingConditionOperator} implementation under the specified operator name.
     * If an operator with the same name already exists, it will be replaced.
     * <p>
     * This allows adding custom condition operators
     * or overriding existing ones (e.g., "greater_than", "in", etc.).
     *
     * @param operatorName the unique name of the operator (e.g., "in", "custom_operator"); must not be {@code null}
     * @param operator     the operator implementation to associate with the name; must not be {@code null}
     */
    public static void register(final String operatorName, final SourcingConditionOperator operator) {

        typeToOperatorMap.put(operatorName, operator);
    }
}
