/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;

/**
 * Represents a generic operator used to evaluate a sourcing condition.
 *
 * <p>Implementations of this interface define specific logic for comparing or checking
 * one two or more operands, such as equality, range checks, membership, or existence.</p>
 *
 * <p>This abstraction allows different operator types to be plugged into the condition
 * evaluation process dynamically based on the configured operator name.</p>
 */
public interface SourcingConditionOperator {

    /**
     * Evaluates the condition using the provided operands.
     *
     * <p>This method supports both unary and binary operators depending on the implementation.
     * The meaning and expected structure of operands varies by operator type:</p>
     *
     * <ul>
     *     <li><b>EQUALS</b> — expects two operands:
     *         <ul>
     *             <li>first: value extracted from the sourcing context (e.g., a field from JSON)</li>
     *             <li>second: reference value to compare against (e.g., a string, number, or boolean)</li>
     *         </ul>
     *     </li>
     *     <li><b>BETWEEN</b> — expects two operands:
     *         <ul>
     *             <li>first: value from the context (must be comparable, e.g., number or date)</li>
     *             <li>second: a {@code Map<String, Object>} with two entries: {@code "from"} and {@code "to"}</li>
     *         </ul>
     *     </li>
     *     <li><b>EXISTS</b> — may expect a single operand</li>
     * </ul>
     *
     * @param operands the input values to evaluate; must not be {@code null}
     * @return {@code true} if the evaluation passes according to the operator's logic,
     * {@code false} otherwise
     * @throws IllegalArgumentException if the number or structure of operands is invalid for the operator
     */
    boolean evaluate(Object... operands);
}
