/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;

/**
 * Operator that evaluates whether a context value is a member of a given list of values.
 *
 * <p>This operator supports the "IN" and "NOT IN" conditions by testing membership of
 * the context value in a list provided as the second operand.</p>
 *
 * <p>Operands:
 * <ol>
 *   <li>The first operand is the context value to check for membership.</li>
 *   <li>The second operand is a {@link List} of values to test membership against.</li>
 * </ol>
 * The context value and list values are converted to a common type before comparison.</p>
 */
public class SourcingConditionMembershipOperator extends AbstractSourcingConditionOperator {

    private final Predicate<Boolean> predicate;

    public SourcingConditionMembershipOperator(Predicate<Boolean> predicate) {
        this.predicate = predicate;
    }

    @Override
    protected boolean doEvaluate(Object... operands) {

        Object contextValue = getContextValue(operands);

        if (contextValue == null || operands[1] == null) {
            return false;
        }

        List<Object> values = new ArrayList<>();

        for (Object value : (List<?>) operands[1]) {
            values.add(convertValue(value, getContextValueType(operands)));
        }

        return predicate.test(values.contains(getContextValue(operands)));
    }


    @Override
    protected boolean validateOperands(Object[] operands) {

        if (operands.length != BINARY_OPERATOR_OPERAND_COUNT) {
            return false;
        }

        return operands[1] == null || operands[1] instanceof List;
    }

    public static boolean ifValueIn(Object... operands) {
        return new SourcingConditionMembershipOperator(result -> result).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueNotIn(Object... operands) {
        return new SourcingConditionMembershipOperator(result -> !result).evaluate(operands[0], operands[1]);
    }
}
