/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;

import java.util.function.Predicate;

/**
 * Implementation of {@link SourcingConditionOperator} that performs mathematical
 * comparisons between a context value and a reference value using a provided predicate.
 *
 * <p>This operator supports binary comparison operators such as GREATER_THAN_OR_EQUALS, LESS_THAN_OR_EQUALS,
 * GREATER_THAN, LESS_THAN, and their variants by applying a {@link Predicate<Integer>}
 * to the result of comparing two {@link Comparable} operands.</p>
 *
 * <p>The first operand is the context value, and the second operand is the value to compare against.
 * Both operands must be {@link Comparable} or {@code null}.</p>
 */
public class SourcingConditionMathOperator extends AbstractSourcingConditionOperator {

    private final Predicate<Integer> predicate;

    public SourcingConditionMathOperator(Predicate<Integer> predicate) {
        this.predicate = predicate;
    }

    @Override
    protected boolean doEvaluate(Object... operands) {

        Object contextValue = getContextValue(operands);

        if (contextValue == null) {
            return false;
        }

        Object value = convertValue(operands[1], getContextValueType(operands));

        if (value == null) {
            return false;
        }

        return predicate.test((toComparable(contextValue)).compareTo(toComparable(value)));
    }


    @Override
    protected boolean validateOperands(Object[] operands) {

        if (operands.length != BINARY_OPERATOR_OPERAND_COUNT) {
            return false;
        }

        Class<?> valueType = getContextValueType(operands);

        return valueType == null || Comparable.class.isAssignableFrom(valueType);
    }

    public static boolean ifValueEquals(Object... operands) {
        return new SourcingConditionMathOperator(compareResult -> compareResult == 0).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueNotEquals(Object... operands) {
        return new SourcingConditionMathOperator(compareResult -> compareResult != 0).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueIsGreaterThan(Object... operands) {
        return new SourcingConditionMathOperator(compareResult -> compareResult > 0).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueIsGreaterThanOrEquals(Object... operands) {
        return new SourcingConditionMathOperator(compareResult -> compareResult >= 0).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueIsLessThan(Object... operands) {
        return new SourcingConditionMathOperator(compareResult -> compareResult < 0).evaluate(operands[0], operands[1]);
    }

    public static boolean ifValueIsLessThanOrEquals(Object... operands) {
        return new SourcingConditionMathOperator(a -> a <= 0).evaluate(operands[0], operands[1]);
    }
}