/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;

public class SourcingConditionExistsOperator extends AbstractSourcingConditionOperator {


    /**
     * Operator that evaluates the existence of a value in the sourcing context.
     *
     * <p>This operator expects exactly two operands:
     * <ol>
     *   <li>The first operand is the context value, which can be any object.</li>
     *   <li>The second operand is a Boolean indicating whether existence is required
     *       ({@code true}) or non-existence is required ({@code false}).</li>
     * </ol>
     * The evaluation returns {@code true} if the presence (non-nullness) of the context value
     * matches the expected Boolean value.</p>
     */
    @Override
    protected boolean doEvaluate(Object... operands) {

        Object contextValue = getContextValue(operands);

        Boolean contextBooleanValue = contextValue != null;

        Object value = convertValue(operands[1], Boolean.class);

        if (value == null) {
            return false;
        }

        return contextBooleanValue.equals(value);
    }


    @Override
    protected boolean validateOperands(Object[] operands) {

        return operands.length == BINARY_OPERATOR_OPERAND_COUNT;
    }

    public static boolean ifExists(Object... operands) {
        return new SourcingConditionExistsOperator().evaluate(operands[0], operands[1]);
    }
}