/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing.condition.operator;


import com.fasterxml.jackson.databind.ObjectMapper;

import java.time.Instant;
import java.time.format.DateTimeParseException;

/**
 * Abstract base class for implementations of {@link SourcingConditionOperator}.
 *
 * <p>This class provides common logic for validating operands and converting values
 * during evaluation of sourcing condition operators. It handles unary and binary operators
 * by defining a contract for subclasses to implement the specific evaluation and validation logic.</p>
 *
 * <p>It also includes a utility method to convert operand values to comparable types,
 * supporting special handling for date/time strings formatted as ISO 8601.</p>
 */
public abstract class AbstractSourcingConditionOperator implements SourcingConditionOperator {

    protected static final int BINARY_OPERATOR_OPERAND_COUNT = 2;


    protected static final ObjectMapper mapper = new ObjectMapper();

    /**
     * Evaluates the operator against the provided operands.
     *
     * <p>This method first validates the operands by calling {@link #validateOperands(Object...)},
     * then delegates the actual evaluation logic to {@link #doEvaluate(Object...)}.</p>
     *
     * @param operands the operands to evaluate; expected to be non-null and conforming to validation rules
     * @return {@code true} if the evaluation passes; {@code false} otherwise
     * @throws IllegalArgumentException if operands are invalid
     */
    @Override
    public boolean evaluate(Object... operands) {

        if (!validateOperands(operands)) {
            throw new IllegalArgumentException("Invalid operands provided.");
        }

        return doEvaluate(operands);
    }

    protected Object getContextValue(Object... operands) {
        return operands[0];
    }

    protected Class<?> getContextValueType(Object... operands) {

        Object value = getContextValue(operands);
        return value == null ? null : value.getClass();
    }

    protected <T extends Comparable<?>> T toComparable(Object operand) {
        return (T) operand;
    }


    protected abstract boolean doEvaluate(Object... operands);

    protected abstract boolean validateOperands(Object... operands);

    protected Object convertValue(Object value, Class<?> type) {

        if (value == null) {
            return null;
        }
        // if context value type is Long then probably value can represent date yyyy-MM-dd'T'HH:mm:ss.SSS'Z' -ISO 8601 UTC.
        // try to convert to instant.
        // if not then it is not a date, try to convert to requested type
        if (Long.class == type) {
            try {
                Instant instant = Instant.parse(mapper.convertValue(value, String.class));
                return instant.toEpochMilli();
            } catch (DateTimeParseException ignored) {

            }
        }

        return mapper.convertValue(value, type);
    }
}
