/*
 * Copyright © 2024, 2025 Fluent Commerce - All Rights Reserved.
 */
package com.fluentcommerce.util.sourcing;

import com.fluentcommerce.graphql.sourcing.queries.location.GetFullLocationByRefQuery;
import com.fluentcommerce.graphql.sourcing.queries.location.GetFullLocationsByNetworkQuery;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.sourcing.context.model.Address;
import com.fluentcommerce.util.sourcing.context.model.Location;
import com.fluentcommerce.util.sourcing.context.model.OpeningSchedule;
import com.fluentcommerce.util.sourcing.context.model.Retailer;
import com.fluentcommerce.util.units.DistanceMeasurementUnits;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import lombok.SneakyThrows;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.sourcing.context.SourcingContextUtils.mapAttributes;

/**
 * Utility class that provides helper methods for working with Location objects.
 *
 * <p>This class is not intended to be instantiated and contains only static utility methods
 * used during sourcing rule execution.</p>
 *
 * <p>This class provides methods to load individual locations by reference,
 * as well as all locations within one or multiple networks. To improve performance
 * and reduce the load on the backend, it uses two internal caches</p>
 *
 * <ul>
 *   <li>{@link #locationCache} caches individual {@link Location} objects by their unique reference.</li>
 *   <li>{@link #networkCache} caches lists of {@link Location} objects keyed by
 *       network references. This avoids repeated expensive queries when fetching
 *       all locations for the same network multiple times.</li>
 * </ul>
 *
 * <p>Both caches are configured with:</p>
 * <ul>
 *   <li>A maximum size of {@value #CACHE_MAX_SIZE} entries to limit memory usage.
 *       When the cache reaches this size, the least recently used entries
 *       are automatically evicted.</li>
 *   <li>An expiration time of {@value #CACHE_EXPIRY_MINUTES} minutes after write,
 *       ensuring cached data is refreshed periodically to keep it reasonably up to date.</li>
 * </ul>
 *
 * <p>This caching strategy strikes a balance between reducing backend query load
 * and maintaining reasonably fresh data for consumers of this utility.</p>
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class LocationUtils {

    private static final int CACHE_EXPIRY_MINUTES = 10;
    private static final int CACHE_MAX_SIZE = 100;

    private static final Cache<String, Optional<Location>> locationCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_EXPIRY_MINUTES, TimeUnit.MINUTES)
            .maximumSize(CACHE_MAX_SIZE)
            .build();

    private static final Cache<String, List<Location>> networkCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_EXPIRY_MINUTES, TimeUnit.MINUTES)
            .maximumSize(CACHE_MAX_SIZE)
            .build();

    /**
     * Resets Location Cache between unit tests to avoid problems related to unnecessary stubbing
     */
    public static void resetCacheForTest() {
        locationCache.invalidateAll();
        networkCache.invalidateAll();
    }

    /**
     * Load all the locations in a network.
     *
     * <p>This method caches the result for each networkRef for {@value #CACHE_EXPIRY_MINUTES} minutes
     * to reduce repeated backend queries. The cache can hold up to {@value #CACHE_MAX_SIZE} entries,
     * and evicts least recently used entries when full.</p>
     *
     * @param context    rule context object.
     * @param networkRef ref of the network to load
     * @return list of Location objects belonging to the network
     */
    @SneakyThrows
    public static List<Location> getLocationsInNetwork(final Context context, final String networkRef) {
        return networkCache.get(networkRef, () -> {
            final GetFullLocationsByNetworkQuery.Data response = (GetFullLocationsByNetworkQuery.Data)
                    context.api().query(GetFullLocationsByNetworkQuery.builder()
                            .network(networkRef)
                            .build());

            ImmutableList.Builder<Location> builder = ImmutableList.builder();
            if (response.network() != null && response.network().locations() != null
                    && response.network().locations().edges() != null) {
                response.network().locations().edges().stream().forEach(edge -> {
                    GetFullLocationsByNetworkQuery.Node loc = edge.node();
                    if (loc != null) {
                        builder.add(mapLocation(loc));
                    }
                });
            }
            return builder.build();
        });
    }

    /**
     * Loads all locations that belong to the provided networks. Networks are loaded one by one.
     *
     * <p>Each network's locations are individually cached as in {@link #getLocationsInNetwork(Context, String)}.
     * This method aggregates those cached or freshly loaded locations into a single list.</p>
     *
     * @param context     rule context object
     * @param networkRefs refs of the networks to load
     * @return combined list of Location objects from all specified networks
     */
    public static List<Location> getLocationsInNetworks(final Context context, final List<String> networkRefs) {
        final ImmutableList.Builder<Location> allLocations = ImmutableList.builder();
        for (String network : networkRefs) {
            final List<Location> locations = LocationUtils.getLocationsInNetwork(context, network);
            allLocations.addAll(locations);
        }
        return allLocations.build();
    }

    /**
     * Loads a single Location by provided ref, caching the results
     *
     * @param context     rule context object
     * @param locationRef ref of the location to load
     * @return the loaded location
     */
    @SneakyThrows
    public static Location getLocationByRef(final Context context, final String locationRef) {
        if (locationRef == null) {
            return null;
        }
        return locationCache.get(locationRef, () -> Optional.of(locationRef)
                .map(ref -> (GetFullLocationByRefQuery.Data)
                        context.api().query(GetFullLocationByRefQuery.builder()
                                .locationRef(ref)
                                .build()))
                .map(GetFullLocationByRefQuery.Data::location)
                .map(LocationUtils::mapLocation)
        ).orElse(null);
    }

    /**
     * Calculate distance between two points in latitude and longitude.
     * [from: <a href="https://stackoverflow.com/questions/3694380/calculating-distance-between-two-points-using-latitude-longitude">...</a>]
     *
     * @param lat1 latitude of the first point
     * @param lon1 longitude of the first point
     * @param lat2 latitude of the second point
     * @param lon2 longitude of the second point
     * @return Distance in Metres
     */
    public static double distanceInMetres(double lat1, double lon1, double lat2, double lon2) {

        return distanceInKilometres(lat1, lon1, lat2, lon2) * 1000;
    }

    /**
     * Calculate distance between two points in latitude and longitude.
     *
     * @param lat1 latitude of the first point
     * @param lon1 longitude of the first point
     * @param lat2 latitude of the second point
     * @param lon2 longitude of the second point
     * @return Distance in Kilometres
     */
    public static double distanceInKilometres(final double lat1, final double lon1, final double lat2, final double lon2) {
        final int radius = 6371; // Radius of the earth in kilometres

        return radius * calculateAngularDistance(lat1, lon1, lat2, lon2);
    }

    /**
     * Calculate distance between two points in latitude and longitude.
     *
     * @param lat1 latitude of the first point
     * @param lon1 longitude of the first point
     * @param lat2 latitude of the second point
     * @param lon2 longitude of the second point
     * @return Distance in Miles
     */
    public static double distanceInMiles(final double lat1, final double lon1, final double lat2, final double lon2) {
        final int radius = 3959; // Radius of the earth in miles

        return radius * calculateAngularDistance(lat1, lon1, lat2, lon2);
    }

    /**
     * Calculate distance between two points using the specified distance unit.
     * 
     * @param lat1 latitude of the first point
     * @param lon1 longitude of the first point
     * @param lat2 latitude of the second point
     * @param lon2 longitude of the second point
     * @param unit distance measurement unit (MILES, KILOMETRES, or METRES)
     * @return distance between the two points in the specified unit
     */
    public static double calculateDistance(final double lat1, final double lon1, 
                                         final double lat2, final double lon2, 
                                         final DistanceMeasurementUnits unit) {

        switch (unit) {
            case MILES:
                return distanceInMiles(lat1, lon1, lat2, lon2);
            case METRES:
                return distanceInMetres(lat1, lon1, lat2, lon2);
            case KILOMETRES:
                return distanceInKilometres(lat1, lon1, lat2, lon2);
            default:
                throw new IllegalArgumentException("Unsupported distance unit: " + unit);
        }
    }

    private static double calculateAngularDistance(final double lat1, final double lon1, final double lat2, final double lon2) {

        double latDistance = Math.toRadians(lat2 - lat1);
        double lonDistance = Math.toRadians(lon2 - lon1);
        double a = Math.sin(latDistance / 2) * Math.sin(latDistance / 2)
                + Math.cos(Math.toRadians(lat1)) * Math.cos(Math.toRadians(lat2))
                * Math.sin(lonDistance / 2) * Math.sin(lonDistance / 2);
        return 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    }

    private static Location mapLocation(GetFullLocationByRefQuery.Location location) {
        if (location == null) {
            return null;
        }
        return Location.builder()
                .id(location.id())
                .ref(location.ref())
                .createdOn((Date) location.createdOn())
                .updatedOn((Date) location.updatedOn())
                .type(location.type())
                .status(location.status())
                .attributes(mapAttributes(location.attributes()))
                .name(location.name())
                .supportPhoneNumber(location.supportPhoneNumber())
                .defaultCarrier(location.defaultCarrier())
                .defaultCarrierName(location.defaultCarrierName())
                .primaryAddress(JsonUtils.anyToPojo(location.primaryAddress(), Address.class))
                .retailer(JsonUtils.anyToPojo(location.retailer(), Retailer.class))
                .networks(Optional.ofNullable(location.networks())
                        .map(GetFullLocationByRefQuery.Networks::edges)
                        .map(edges -> edges.stream()
                                .map(edge -> Optional.ofNullable(edge)
                                        .map(GetFullLocationByRefQuery.Edge::node)
                                        .map(GetFullLocationByRefQuery.Node::ref)
                                        .orElse(null))
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList()))
                        .orElse(null))
                .openingSchedule(JsonUtils.anyToPojo(location.openingSchedule(), OpeningSchedule.class))
                .build();
    }

    private static Location mapLocation(GetFullLocationsByNetworkQuery.Node location) {
        if (location == null) {
            return null;
        }
        return Location.builder()
                .id(location.id())
                .ref(location.ref())
                .createdOn((Date) location.createdOn())
                .updatedOn((Date) location.updatedOn())
                .type(location.type())
                .status(location.status())
                .attributes(mapAttributes(location.attributes()))
                .name(location.name())
                .supportPhoneNumber(location.supportPhoneNumber())
                .defaultCarrier(location.defaultCarrier())
                .defaultCarrierName(location.defaultCarrierName())
                .primaryAddress(JsonUtils.anyToPojo(location.primaryAddress(), Address.class))
                .retailer(JsonUtils.anyToPojo(location.retailer(), Retailer.class))
                .networks(Optional.ofNullable(location.networks())
                        .map(GetFullLocationsByNetworkQuery.Networks::edges)
                        .map(edges -> edges.stream()
                                .map(edge -> Optional.ofNullable(edge)
                                        .map(GetFullLocationsByNetworkQuery.Edge1::node)
                                        .map(GetFullLocationsByNetworkQuery.Node1::ref)
                                        .orElse(null))
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList()))
                        .orElse(null))
                .openingSchedule(JsonUtils.anyToPojo(location.openingSchedule(), OpeningSchedule.class))
                .build();
    }

}
