package com.fluentcommerce.util.dynamic.graphql;

import com.fasterxml.jackson.module.jsonSchema.JsonSchema;
import com.fasterxml.jackson.module.jsonSchema.types.ReferenceSchema;
import com.fluentcommerce.graphql.queries.introspection.IntrospectionQuery;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
public class GraphQLIntrospectionUtils {

    private static IntrospectionQuery.Data introspectionData = null;

    public static IntrospectionQuery.Data getSchema(Context context) {
        if(introspectionData == null) {
            introspectionData = (IntrospectionQuery.Data) context.api().query(IntrospectionQuery.builder().build());
        }
        return introspectionData;
    }

    public static boolean isSchemaLoaded() {
        return introspectionData != null;
    }

    private static final Map<String, List<QueryParameter>> queryArgs = new HashMap<>();

    public static List<QueryParameter> getArgsForQuery(Context context, String queryName) {
        if(!queryArgs.containsKey(queryName)) {
            getQuery(context, queryName).ifPresent(q -> {
                queryArgs.put(queryName, q.args().stream()
                        .map(GraphQLIntrospectionUtils::argToQueryParam)
                        .collect(Collectors.toList()));
            });
        }
        return queryArgs.get(queryName);
    }

    private static final Map<String, Optional<IntrospectionQuery.Type>> queryTypes = new HashMap<String, Optional<IntrospectionQuery.Type>>();

    public static Optional<IntrospectionQuery.Type> getResponseTypeForQuery(Context context, String queryName) {
        if(!queryTypes.containsKey(queryName)) {

            getQuery(context, queryName).ifPresent(q -> {
                queryTypes.put(queryName,
                        getSchema(context).__schema().types().stream().filter(t -> t.name().equals(q.type().name())).findFirst()
                );
            });
        }
        return queryTypes.get(queryName);
    }

    public static Optional<IntrospectionQuery.Field> getField(IntrospectionQuery.Type type, String fieldName) {
        return type.fields().stream().filter((f) -> fieldName.equals(f.name())).findFirst();
    }


    /**
     * Find input fields for inputName from schema.
     * @param context - ruleSet context.
     * @param inputName - name of input object.
     * @return
     */
    public static List<IntrospectionQuery.InputField> getFieldsForInput(Context context, String inputName) {
        Optional<IntrospectionQuery.Type> foundType = getSchema(context).__schema()
                .types().stream().filter(type -> type.name().equals(inputName)).findFirst();
        if (foundType.isPresent()) {
            return foundType.get().inputFields();
        }
        return Collections.emptyList();
    }

    public static Optional<IntrospectionQuery.Type> getResponseTypeForField(Context context, IntrospectionQuery.Field field) {
        IntrospectionQuery.Type2 type = field.type();
        String typeName = type.name();
        if(type.ofType() != null) {
            typeName = type.ofType().name();
            if(type.ofType().ofType() != null) {
                typeName = type.ofType().ofType().name();
                if(type.ofType().ofType().ofType() != null) {
                    typeName = type.ofType().ofType().ofType().name();
                    if (type.ofType().ofType().ofType().ofType() != null) {
                        typeName = type.ofType().ofType().ofType().ofType().name();
                    }
                }
            }
        }

        final String finalTypeName = typeName;
        return getSchema(context).__schema().types().stream()
                .filter(t -> finalTypeName.equals(t.name())).findFirst();
    }

    /**
     * Get response type for argument of query, e.g. order(id:ID, ref:String), return type for "ref"
     * @param context - ruleSet context.
     * @param field - argument, e.g. ref
     * @return - type of argument, e.g. String
     */
    public static Optional<IntrospectionQuery.Type> getResponseTypeForArg(Context context, IntrospectionQuery.Arg field) {
        IntrospectionQuery.Type1 type = field.type();
        String typeName = type.name();
        if(type.ofType() != null) {
            typeName = type.ofType().name();
            if(type.ofType().ofType() != null) {
                typeName = type.ofType().ofType().name();
                if(type.ofType().ofType().ofType() != null) {
                    typeName = type.ofType().ofType().ofType().name();
                    if (type.ofType().ofType().ofType().ofType() != null) {
                        typeName = type.ofType().ofType().ofType().ofType().name();
                    }
                }
            }
        }

        final String finalTypeName = typeName;
        return getSchema(context).__schema().types().stream()
                .filter(t -> finalTypeName.equals(t.name())).findFirst();
    }

    private static final Map<String, Optional<IntrospectionQuery.Field>> queries = new HashMap<>();

    /**
     * Returns IntrospectionQuery.Field from schema which is corresponds to queryName.
     * e.g. "order.fulfilmentChoices.items" -> order() { fulfilmentChoices { edges { node { items } } } }
     *
     * @param context
     * @param queryName
     * @return
     */
    public static Optional<IntrospectionQuery.Field> getQuery(Context context, String queryName) {
        if(!queries.containsKey(queryName)) {
            final Optional<IntrospectionQuery.Type> rootQuery = getRootQuery(context);
            final Optional<IntrospectionQuery.Field> resultQuery = findQuery(queryName, context, rootQuery);
            queries.put(queryName, resultQuery);
        }
        return queries.get(queryName);
    }

    /**
     * Get input field if it is exist in query parameter form.
     * @param context - ruleSet context.
     * @param path - path to input field, e.g. order.fulfilmentChoices.items.createdOn.from
     * @return - query parameter.
     */
    public static Optional<QueryParameter> getInputFieldLikeQueryParam(Context context, String path) {
        final Optional<IntrospectionQuery.Type> rootQuery = getRootQuery(context);
        return findInputParameterLikeQueryParam(path, context, rootQuery);
    }

    private static Optional<IntrospectionQuery.Type> getRootQuery(Context context) {
        return getSchema(context).__schema().types().stream()
                .filter(type -> "Query".equalsIgnoreCase(type.name())).findFirst();
    }

    private static Optional<QueryParameter> findInputParameterLikeQueryParam(String path, Context context, Optional<IntrospectionQuery.Type> rootQuery) {
        if (!rootQuery.isPresent()) {
            return Optional.empty();
        }
        final List<String> pathParts = Arrays.asList(path.split("\\."));

        Optional<IntrospectionQuery.Field> query = Optional.empty();
        Optional<IntrospectionQuery.Type> queryType = rootQuery;
        Optional<IntrospectionQuery.Type> inputFieldType = Optional.empty();
        Optional<IntrospectionQuery.Arg> inputArgument = Optional.empty();

        for(int i = 0; i < pathParts.size(); i++) {
            final String part = pathParts.get(i);

            if (inputArgument.isPresent()) {
                inputFieldType = getResponseTypeForArg(context, inputArgument.get());
                if (inputFieldType.isPresent()) {
                    Optional<IntrospectionQuery.InputField> inputField =
                            inputFieldType.get().inputFields().stream().filter(f -> part.equals(f.name())).findFirst();
                    if (inputField.isPresent()) {
                        return Optional.of(inputFieldToQueryParam(inputField.get()));
                    } else {
                        log.warn("Can't find Input parameter for path {}", path);
                        return Optional.empty();
                    }

                }
            }

            if (!inputFieldType.isPresent()) {
                if (query.isPresent()) {
                    inputArgument = query
                            .get()
                            .args()
                            .stream()
                            .filter(arg -> StringUtils.equals(arg.name(), part))
                            .findFirst();
                }
                query = findQuery(part, context, queryType);
                if (query.isPresent()) {
                    queryType = getResponseTypeForField(context, query.get());
                } else {
                    if (!inputArgument.isPresent()) {
                        log.warn("Can't find Input parameter and query for path {}", path);
                    }
                }
            }

        }

        if (!inputFieldType.isPresent()) {
            if (inputArgument.isPresent()) {
                return Optional.of(argToQueryParam(inputArgument.get()));
            }
        }

        log.warn("Can't find Input parameter for path {}", path);
        return Optional.empty();
    }

    private static Optional<IntrospectionQuery.Field> findQuery(String queryName,
                                                                Context context,
                                                                Optional<IntrospectionQuery.Type> rootQuery) {
        if (!rootQuery.isPresent()) {
            return Optional.empty();
        }
        final List<String> queryParts = new ArrayList<>(Arrays.asList(queryName.split("\\.")));
        if (queryParts.size() <= 1) {
            if (StringUtils.endsWith(rootQuery.get().name(), "Connection")) {
                return findQuery( "edges.node." + queryName, context, rootQuery);
            }
            return rootQuery
                    .map(IntrospectionQuery.Type::fields)
                    .filter(Objects::nonNull)
                    .flatMap(fields -> fields.stream().filter(field -> field.name().equals(queryName)).findFirst());
        }
        Optional<IntrospectionQuery.Field> subQuery = Optional.empty();
        Optional<IntrospectionQuery.Type> subQueryType = rootQuery;

        for (int i = 0; i < queryParts.size(); i++) {
            final String part = queryParts.get(i);
            subQuery = getField(subQueryType.get(), part);
            if (!subQuery.isPresent()) {
                return Optional.empty();
            }
            subQueryType = getResponseTypeForField(context, subQuery.get());
            if (!subQueryType.isPresent()) {
                return Optional.empty();
            }
            if (StringUtils.endsWith(subQueryType.get().name(), "Connection") && (i + 1 < queryParts.size())) {
                queryParts.add(i + 1, "edges");
                queryParts.add(i + 2, "node");
            }
        }

        return subQuery;
    }


    private static QueryParameter argToQueryParam(IntrospectionQuery.Arg arg) {
        /*
            This abomination was brought to you by GraphQLs weird type metastructure in conjunction with Apollo's
            terrible class generation. GraphQL types can be nested indefinitely to "decorate" raw types, so an arg like
            `thing:[String!]!` would be defined something like:
                {name:null kind:NON_NULL ofType:
                    {name:null kind:LIST ofType:
                        {name:null kind:NON_NULL ofType:
                            {name:String}}}}

            I think this does the job for most of our types but could use some work to properly represent some edge
             cases like nested lists?
         */
        IntrospectionQuery.Type1 type = arg.type();
        String stringType = type.name();
        boolean isMandatory = type.kind().name().equals("NON_NULL");
        boolean isList = type.kind().name().equals("LIST");

        if(type.ofType() != null) {
            IntrospectionQuery.OfType ofType = type.ofType();
            stringType = ofType.name();
            isMandatory = isMandatory || ofType.kind().name().equals("NON_NULL");
            isList = isList || (isMandatory && ofType.kind().name().equals("LIST"));

            if(ofType.ofType() != null) {
                IntrospectionQuery.OfType1 ofType1 = ofType.ofType();
                stringType = ofType1.name();
                isMandatory = isMandatory || ofType1.kind().name().equals("NON_NULL");
                isList = isList || ofType1.kind().name().equals("LIST");

                if(ofType1.ofType() != null) {
                    IntrospectionQuery.OfType2 ofType2 = ofType1.ofType();
                    stringType = ofType2.name();
                    isMandatory = isMandatory || ofType2.kind().name().equals("NON_NULL");
                    isList = isList || ofType2.kind().name().equals("LIST");

                    if(ofType2.ofType() != null) {
                        IntrospectionQuery.OfType3 ofType3 = ofType2.ofType();
                        stringType = ofType3.name();
                        isMandatory = isMandatory || ofType3.kind().name().equals("NON_NULL");
                        isList = isList || ofType3.kind().name().equals("LIST");
                    }
                }
            }
        }

        return new QueryParameter(arg.name(), stringType, isMandatory, isList);
    }

    private static Map<String, List<QueryParameter>> mutationFields = new HashMap<>();

    public static List<QueryParameter> getFieldsForMutation(Context context, String inputName) {
        if(!mutationFields.containsKey(inputName)) {
            IntrospectionQuery.Data intro = getSchema(context);

            Optional<IntrospectionQuery.Type> inputType = intro.__schema().types().stream()
                    .filter(type -> type.name().equals(inputName))
                    .findFirst();

            inputType.ifPresent(in -> {
                List<QueryParameter> fields = in.inputFields().stream()
                        .map(GraphQLIntrospectionUtils::inputFieldToQueryParam)
                        .collect(Collectors.toList());

                mutationFields.put(inputName, fields);
            });
        }

        return mutationFields.get(inputName);
    }


    private static QueryParameter inputFieldToQueryParam(IntrospectionQuery.InputField field) {
        IntrospectionQuery.Type3 type = field.type();
        String stringType = type.name();
        boolean isMandatory = type.kind().name().equals("NON_NULL");
        boolean isList = type.kind().name().equals("LIST");

        if(type.ofType() != null) {
            IntrospectionQuery.OfType8 ofType = type.ofType();
            stringType = ofType.name();
            isMandatory = isMandatory || ofType.kind().name().equals("NON_NULL");
            isList = isList || ofType.kind().name().equals("LIST");

            if(ofType.ofType() != null) {
                IntrospectionQuery.OfType9 ofType1 = ofType.ofType();
                stringType = ofType1.name();
                isMandatory = isMandatory || ofType1.kind().name().equals("NON_NULL");
                isList = isList || ofType1.kind().name().equals("LIST");

                if(ofType1.ofType() != null) {
                    IntrospectionQuery.OfType10 ofType2 = ofType1.ofType();
                    stringType = ofType2.name();
                    isMandatory = isMandatory || ofType2.kind().name().equals("NON_NULL");
                    isList = isList || ofType2.kind().name().equals("LIST");

                    if(ofType2.ofType() != null) {
                        IntrospectionQuery.OfType11 ofType3 = ofType2.ofType();
                        stringType = ofType3.name();
                        isMandatory = isMandatory || ofType3.kind().name().equals("NON_NULL");
                        isList = isList || ofType3.kind().name().equals("LIST");
                    }
                }
            }
        }
        return new QueryParameter(field.name(), stringType, isMandatory, isList);
    }

    /**
     * Get list of fields for path within json.
     * @param node - e.g. { "retailer": {"id": 123, "status": "CREATED"} }
     * @param path - e.g. "retailer"
     * @param allSchemas - json schemas for all objects from @param node, it requires to resolve ReferenceSchemas coming
     *                  from jackson schema generating mechanism.
     * @return list of fields: ["retailer.id", "retailer.status"]
     */
    public static List<String> deriveFieldsFromJSON(Map<String, JsonSchema> allSchemas, JsonSchema node, String path) {
        if (node.isObjectSchema() && !node.asObjectSchema().getProperties().isEmpty()) {
            return node.asObjectSchema().getProperties().entrySet().stream()
                    // skip inner connections, they need processing separately
                    .filter(e -> !e.getValue().isArraySchema() || StringUtils.contains(e.getKey(), "attributes"))
                    .flatMap(e -> {
                        final String nextPath = path + (path.length() > 0 ? "." : "") + e.getKey();
                        if (e.getValue() instanceof ReferenceSchema) { // resolve reference object
                            JsonSchema schema = allSchemas.get(e.getValue().get$ref());
                            if (schema != null) {
                                return deriveFieldsFromJSON(allSchemas, schema, nextPath).stream();
                            } else {
                                log.warn("Can't resolve reference {}, urn {}", nextPath, e.getValue().get$ref());
                                return Stream.empty();
                            }
                        } else {
                            return deriveFieldsFromJSON(allSchemas, e.getValue(), nextPath).stream();
                        }
                    })
                    .collect(Collectors.toList());
        }
        if (node.isArraySchema()) {
            return deriveFieldsFromJSON(allSchemas, node.asArraySchema().getItems().asSingleItems().getSchema(), path);
        }

        if (node instanceof ReferenceSchema) {
            JsonSchema schema = allSchemas.get(node.get$ref());
            if (schema != null) {
                return deriveFieldsFromJSON(allSchemas, schema, path);
            } else {
                log.warn("Can't resolve reference {}, urn {}", path, node.get$ref());
            }
        }
        return ImmutableList.of(path);
    };

    /**
     * Get list of fields for path within json. Try to collect and resolve all reference schemas.
     * @param node - e.g. { "retailer": {"id": 123, "status": "CREATED"} }
     * @param path - e.g. "retailer"
     * @return list of fields: ["retailer.id", "retailer.status"]
     */
    public static List<String> deriveFieldsFromJSON(JsonSchema node, String path) {
        Map<String, JsonSchema> schemaList = JsonUtils.collectAllSchemas(node);
        return deriveFieldsFromJSON(schemaList, node, path);
    };

    @Value
    public static final class QueryParameter {
        String name;
        String type;
        boolean required;
        boolean list;

        public String typeDefinition() {
            final StringBuilder graphQlType = new StringBuilder();
            graphQlType.append(type);
            if (required) {
                graphQlType.append("!");
            }
            if (list) {
                graphQlType.insert(0,"[");
                graphQlType.append("]");
            }
            return graphQlType.toString();
        }
    }


    @Value
    public static class CollectedQueryParameter {
        QueryParameter queryParameter;

        String alias;

        String path;

        Object value;
    }
}
