package com.fluentcommerce.util.dynamic.graphql;

import com.apollographql.apollo.api.InputFieldMarshaller;
import com.apollographql.apollo.api.InputFieldWriter;
import com.apollographql.apollo.api.Mutation;
import com.apollographql.apollo.api.OperationName;
import com.apollographql.apollo.api.ResponseField;
import com.apollographql.apollo.api.ResponseFieldMapper;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.base.CaseFormat;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.AbstractMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes.MutationDynamicVariables;
import static com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes.MutationInnerData;
import static com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes.MutationOuterData;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.QueryParameter;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getFieldsForMutation;

public class DynamicUpdateMutation implements Mutation<MutationOuterData, MutationOuterData, MutationDynamicVariables> {

    private static final String RESPONSE_ALIAS = "r";
    private final String queryName;
    private final String inputName;

    private final String queryString;
    private final MutationDynamicVariables variables;
    private final OperationName name;

    private final Boolean errorOnInvalidVariables;

    public DynamicUpdateMutation(Context context, String key, Object value) {
        this(context, ImmutableMap.of(key, value), true);
    }

    public DynamicUpdateMutation(Context context, Object value) {
        this(context, value, false);
    }

    public DynamicUpdateMutation(Context context, Object value, boolean errorOnInvalidVariables) {
        this(context, JsonUtils.pojoToMap(value), errorOnInvalidVariables);
    }

    public DynamicUpdateMutation(Context context, Map<String, Object> variables) {
        this(context, variables, false);
    }

    public DynamicUpdateMutation(Context context, Map<String, Object> variables, boolean errorOnInvalidVariables) {
        this.errorOnInvalidVariables = errorOnInvalidVariables;

        this.queryName = resolveQueryFromContext(context);
        this.inputName = CaseFormat.LOWER_CAMEL.to(CaseFormat.UPPER_CAMEL, this.queryName) + "Input";
        this.queryString = buildQueryString();
        this.variables = resolveVariablesForQuery(queryName, context, variables);
        this.name = new DynamicOperationName(queryName);
    }

    private String resolveQueryFromContext(Context context) {
        String queryName;
        final String type = CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.UPPER_CAMEL, context.getEvent().getEntityType());
        switch(type) {
            case "FulfilmentOptions":
                queryName = "updateFulfilmentOption";
                break;
            default:
                queryName = "update" + type;
        }
        return queryName;
    }

    private String buildQueryString() {
        return String.format("mutation %s($input:%s!) { %s:%s(input:$input) { __typename ref } }", queryName, inputName, RESPONSE_ALIAS, queryName);
    }

    public String getQueryType() {
        return "DynamicUpdateMutation";
    }

    private MutationDynamicVariables resolveVariablesForQuery(String query, Context context, Map<String, Object> passedVariables) {

        // check what fields are present on this input type
        String inputName = CaseFormat.LOWER_CAMEL.to(CaseFormat.UPPER_CAMEL, query) + "Input";
        List<QueryParameter> fields = getFieldsForMutation(context, inputName);

        // derive mandatory variables from context
        try {
            Map<String, Object> derivedVariables = fields.stream()
                    .filter(QueryParameter::isRequired)
                    .map(p -> mapParameter(p, context))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

            // validate that inbound variables belong to input type
            Map<String, String> fieldTypes = fields.stream().collect(
                    Collectors.toMap(QueryParameter::getName, QueryParameter::getType));

            // TODO: single-tier key validation only - support type and nested validation later
            List<String> invalidVars = passedVariables.keySet().stream()
                    .filter(f -> !fieldTypes.containsKey(f))
                    .collect(Collectors.toList());

            if(errorOnInvalidVariables && !invalidVars.isEmpty()) {
                throw new IllegalArgumentException(String.format("Invalid parameters for dynamic update query '%s': %s", queryName, invalidVars));
            }

            invalidVars.forEach(passedVariables::remove);

            return new MutationDynamicVariables(ImmutableMap.of("input", ImmutableMap.<String, Object>builder()
                    .putAll(passedVariables)
                    .putAll(derivedVariables)
                    .build()));

        } catch (Exception e) {
            throw new IllegalArgumentException("Failed to build DynamicQuery for context: " + JsonUtils.objectToNode(context));
        }
    }

    private Map.Entry<String, Object> mapParameter(QueryParameter param, Context context) {
        String name = param.getName();
        Event event = context.getEvent();

        switch(name) {
            case "ref": return new AbstractMap.SimpleEntry<>(name, event.getEntityRef());
            case "id": return new AbstractMap.SimpleEntry<>(name, event.getEntityId());
            case "retailer": return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of("id", event.getRetailerId()));
            case "catalogue": return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of("ref", event.getRootEntityRef()));
            case "returnOrder":
                return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of(
                        "ref", event.getRootEntityRef(),
                        "retailer", ImmutableMap.of("id", event.getRetailerId())
                ));
            default: throw new IllegalArgumentException("Cannot build a dynamic query with parameter: " + name);
        }
    }

    @Override
    public String queryDocument() {
        return queryString;
    }

    @Override
    public MutationDynamicVariables variables() {
        return variables;
    }

    @Override
    public ResponseFieldMapper<MutationOuterData> responseFieldMapper() {
        return readerOuter -> {
            MutationInnerData inner = readerOuter
                    .readObject(ResponseField.forObject(RESPONSE_ALIAS, RESPONSE_ALIAS, null, true, null), reader -> {
                final String __typename = reader.readString(ResponseField.forString("__typename", "__typename", null, false, null));
                final String ref = reader.readString(ResponseField.forString("ref", "ref", null, false, null));
                return new MutationInnerData(__typename, ref);
            });
            return new MutationOuterData(inner);
        };
    }

    @Override
    public MutationOuterData wrapData(MutationOuterData data) {
        return data;
    }

    @Nonnull
    @Override
    public OperationName name() {
        return name;
    }

    @Nonnull
    @Override
    public String operationId() {
        return UUID.randomUUID().toString();
    }

    public static final class MapBasedInputFieldMarshaller implements InputFieldMarshaller {
        private final Map<String, Object> values;

        public MapBasedInputFieldMarshaller(Map<String, Object> values) {
            this.values = values;
        }

        @Override
        public void marshal(InputFieldWriter writer) throws IOException {
            for(Map.Entry<String, Object> e: values.entrySet()) {
                addItem(writer, e.getKey(), e.getValue());
            }
        }

        private void addItem(InputFieldWriter writer, String key, Object value) throws IOException {
            if(value instanceof List) {
                writer.writeList(key, listWriter -> {
                    for (Object v : ((List) value)) {
                        if(v instanceof String) {
                            listWriter.writeString((String) v);
                        } else if(v instanceof Integer) {
                            listWriter.writeInt((Integer) v);
                        } else if(v instanceof Double) {
                            listWriter.writeDouble((Double) v);
                        } else if(v instanceof Boolean) {
                            listWriter.writeBoolean((Boolean) v);
                        } else if(v instanceof Long) {
                            listWriter.writeLong((Long) v);
                        } else if(v instanceof Map) {
                            listWriter.writeObject(new MapBasedInputFieldMarshaller((Map<String, Object>) v));
                        } else {
                            listWriter.writeObject(new MapBasedInputFieldMarshaller(JsonUtils.pojoToMap(v)));
                        }
                    }
                });
            }

            else if(value instanceof String) {
                writer.writeString(key, (String) value);
            } else if(value instanceof Integer) {
                writer.writeInt(key, (Integer) value);
            } else if(value instanceof Double) {
                writer.writeDouble(key, (Double) value);
            } else if(value instanceof Boolean) {
                writer.writeBoolean(key, (Boolean) value);
            } else if(value instanceof Long) {
                writer.writeLong(key, (Long) value);
            } else if(value instanceof Map) {
                writer.writeObject(key, new MapBasedInputFieldMarshaller((Map<String, Object>) value));
            } else {
                writer.writeObject(key, new MapBasedInputFieldMarshaller(JsonUtils.pojoToMap(value)));
            }
        }
    }
}

