package com.fluentcommerce.util.dynamic.graphql;

import com.apollographql.apollo.api.OperationName;
import com.apollographql.apollo.api.Query;
import com.apollographql.apollo.api.ResponseField;
import com.apollographql.apollo.api.ResponseFieldMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.graphql.queries.introspection.IntrospectionQuery;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.dynamic.types.Connection;
import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.graphql.type.CustomType;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.base.CaseFormat;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes.QueryDynamicData;
import static com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes.QueryDynamicVariables;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.CollectedQueryParameter;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.QueryParameter;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.deriveFieldsFromJSON;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getArgsForQuery;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getField;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getFieldsForInput;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getInputFieldLikeQueryParam;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getResponseTypeForField;
import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.getResponseTypeForQuery;

/**
 * DynamicQuery auto-generates a GraphQL/Apollo query for the current entity being processed
 * by a Rule. Query name and primary key are automatically identified from the Rubix Context
 * object (which must be provided at creation).
 *
 * Result type can be provided as a list of paths in JSONPath-like syntax
 * (e.g. ["type","status","attributes.name","attributes.value","attributes.type"]), which can
 * be returned as a JsonNode, or automatically derived from a Pojo and automatically converted
 * directly into that type.
 */
@Slf4j
public class DynamicEntityQuery implements Query<QueryDynamicData, QueryDynamicData, QueryDynamicVariables> {

    private static final String RESPONSE_ALIAS = "r";

    private static final String ARG_VAL_PATTERN = "%s:$%s";
    private static final String ARG_OBJECT_VAL_PATTERN = "%s:{ %s }";
    private final String queryName;

    private final String fullQueryString;
    private final QueryDynamicVariables dynamicVariables;

    private final List<CollectedQueryParameter> collectedQueryParams;
    private final OperationName name;

    /**
     * Create a DynamicQuery using a POJO as the query response type.
     *
     * @param context Ruleset Context used to figure out how to query for the current entity.
     * @param responseType Java POJO from which to derive the response fields
     */
    public DynamicEntityQuery(Context context, Class<?> responseType) {
        this(context, deriveFieldsFromJSON(JsonUtils.getSchemaForClass(responseType), ""));
    }



    /**
     * Create a DynamicQuery using a POJO as the query response type.
     *
     * @param context Ruleset Context used to figure out how to query for the current entity.
     * @param responseType Java POJO from which to derive the response fields
     * @param queryParams list of query params
     */
    public DynamicEntityQuery(Context context, Class<?> responseType, Map<String, Object> queryParams) {
        this(context, deriveFieldsFromJSON(JsonUtils.getSchemaForClass(responseType), ""), null, queryParams);
    }

    /**
     * Create a DynamicQuery with manually defined response paths. Generally the Pojo variant
     * should be preferred over this except for cases where the response values are passed into
     * a Rule as Rule parameters.
     *
     * @param context Ruleset Context used to figure out how to query for the current entity.
     * @param paths List of JSONPath-style selectors, e.g. ["status","fulfilments.edges.node.ref"]
     */
    public DynamicEntityQuery(Context context, List<String> paths) {
       this(context, paths,null, null);
    }


    /**
     *
     * @param context Ruleset Context used to figure out how to query for the current entity.
     * @param paths List of JSONPath-style selectors, e.g. ["status","fulfilments.ref"]
     * @param graphqlQueryName - graphql query name. It is possible to specify otherwise it will be taken from rule context, e.g. "consignmentById".
     * @param queryParameters - list of query parameters,
     *                       e.g. in scope of Order {"ref": "orderRef", "items": {"createdOn": { "from": "2023-12-01T08", "to":"2023-12-01T08" }}}
     */

    public DynamicEntityQuery(Context context, List<String> paths, String graphqlQueryName, Map<String, Object> queryParameters) {
        this.queryName = StringUtils.isEmpty(graphqlQueryName) ? resolveQueryFromContext(context) : graphqlQueryName;
        this.name = () -> this.queryName;
        this.collectedQueryParams = collectQueryParams(queryParameters == null ? new HashMap<>() : queryParameters, queryName, context);

        this.dynamicVariables = resolveVariablesForQuery(queryName, collectedQueryParams, context);

        String argDef = resolveArgDefinitionStructure(queryName, collectedQueryParams, context);
        String argVal = resolveArgValueStructure(queryName, collectedQueryParams, context);
        ResponseTreeNode response = resolveResponseQueryStructure(paths, queryName, context);

        this.fullQueryString = buildQueryString(argDef, argVal, response.toQueryFragment(Collections.singletonList(queryName),
                context, collectedQueryParams));
    }

    private String resolveQueryFromContext(Context context) {
        String queryName;
        switch(context.getEvent().getEntityType()) {
            case "PRODUCT":
                // special case for Product as there's no interface-level update mutation
                if (StringUtils.isEmpty(context.getEvent().getEntitySubtype())) {
                    queryName = CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, context.getEvent().getEntityType());
                } else {
                    queryName = CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, context.getEvent().getEntitySubtype()) + "Product";
                }
                break;
            default:
                queryName = CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, context.getEvent().getEntityType());
        }

        if(getArgsForQuery(context, queryName) == null) {
            // some older entities are only fetchable by ID
            queryName = queryName + "ById";
        }

        return queryName;
    }

    private String buildQueryString(String argDefinition, String argValues, String response) {
        return String.format("query %s(%s) { %s:%s(%s) %s }", queryName, argDefinition, RESPONSE_ALIAS, queryName, argValues, response);
    }

    private String resolveArgDefinitionStructure(String query, List<CollectedQueryParameter> collectedQueryParams, Context context) {
        final BiFunction<String, String, String> toDefinition = (name, type) -> String.format("$%s:%s", name, type);
        Stream<String> argDefinitionFromParameters = collectedQueryParams.stream()
                .map(collectedQueryParameter -> toDefinition.apply(collectedQueryParameter.getAlias(), collectedQueryParameter.getQueryParameter().typeDefinition()));

        Stream<String> argDefinitionFromQuery =  getArgsForQuery(context, query).stream()
                .filter((a) -> {
                    // either argument provided manually in collectedQueryParams or it has to be taken automatically
                    String argPath = query + "." + a.getName();
                    boolean isArgumentInQueryParams = collectedQueryParams.stream()
                            .anyMatch(collectedQueryParameter -> StringUtils.equals(argPath, collectedQueryParameter.getPath()));
                    return  (a.isRequired() || a.getName().equals("id")) && !isArgumentInQueryParams;
                })
                .map(p -> toDefinition.apply(p.getName(), p.typeDefinition()));
        return Stream.concat(argDefinitionFromQuery, argDefinitionFromParameters).collect(Collectors.joining(", "));
    }

    private String resolveArgValueStructure(String query, List<CollectedQueryParameter> collectedQueryParams, Context context) {
        return getArgsForQuery(context, query).stream()
                .filter((a) -> a.isRequired() || a.getName().equals("id"))
                .map(p -> {
                    String argPath = query + "." + p.getName();
                    Optional<CollectedQueryParameter> providedQueryParam = collectedQueryParams.stream()
                            .filter(cqp -> StringUtils.equals(argPath, cqp.getPath())).findFirst();
                    if (providedQueryParam.isPresent()) {
                        // if query parameter provided manually then specify alias as value
                        return String.format(ARG_VAL_PATTERN, p.getName(), providedQueryParam.get().getAlias());
                    }
                   return  String.format(ARG_VAL_PATTERN, p.getName(), p.getName());
                })
                .collect(Collectors.joining(", "));
    }
    private List<GraphQLIntrospectionUtils.CollectedQueryParameter> collectQueryParams(Map<String, Object> queryParameters, String queryName, Context context) {
        final List<GraphQLIntrospectionUtils.CollectedQueryParameter> collectedQueryParameters = new ArrayList<>();
        queryParameters.entrySet().stream()
                .map(keyValue -> this.getParamAliasAndValue(keyValue.getKey(), keyValue.getValue()))
                .flatMap(Collection::stream)
                .forEach(keyValue -> {
                    String paramPath = queryName + "." + keyValue.getKey();
                    String paramAlias = paramPath.replace(".", "_");
                    Optional<QueryParameter> queryParameterFromSchema = getInputFieldLikeQueryParam(context, paramPath);
                    queryParameterFromSchema.ifPresent(q -> collectedQueryParameters.add(new GraphQLIntrospectionUtils.CollectedQueryParameter(q,
                            paramAlias, paramPath, keyValue.getValue())));
                });
        return collectedQueryParameters;
    }

    private List<AbstractMap.SimpleEntry<String, Object>> getParamAliasAndValue(String key, Object value) {
        List<AbstractMap.SimpleEntry<String, Object>> resultKeys = new ArrayList<>();
        if(value instanceof Map) {
            Map<String, Object> valueMap = (Map<String, Object>) value;
            valueMap.forEach((key1, value1) -> {
                List<AbstractMap.SimpleEntry<String, Object>> nextKeys = getParamAliasAndValue(key1, value1);
                nextKeys.forEach(nextKey -> resultKeys.add(new AbstractMap.SimpleEntry<>(key + "." + nextKey.getKey(), nextKey.getValue())));
            });
        } else {
            resultKeys.add(new AbstractMap.SimpleEntry<>(key, value));
        }
        return resultKeys;
    }

    private ResponseTreeNode resolveResponseQueryStructure(List<String> paths, String query, Context context) {
        ResponseTreeNode root = new ResponseTreeNode();

        // stack to track which part of the schema we need to validate against as we iterate
        Optional<IntrospectionQuery.Type> type = getResponseTypeForQuery(context, query);
        if(!type.isPresent()) {
            throw new RuntimeException("problem loading type for query: " + query);
        }

        paths.forEach(path -> {
            try {
                resolvePath(root, type.get(), ImmutableList.copyOf(path.split("\\.")), context);
            } catch(RuntimeException e) {
                log.warn("DynamicEntityQuery failed to add path [{}] to query [{}]", path, queryName);
            }
        });

        return root;
    }

    private void resolvePath(ResponseTreeNode node, IntrospectionQuery.Type parentType, List<String> path, Context context) {
        String segment = path.get(0);

        Optional<IntrospectionQuery.Field> fieldLookup = getField(parentType, segment);

        if(!fieldLookup.isPresent()) {
            // auto-skip connection.edges
            if(getField(parentType, "edges").isPresent()) {
                resolvePath(node, parentType, ImmutableList.<String>builder().add("edges").addAll(path).build(), context);
                if (getField(parentType, "pageInfo").isPresent()) {
                    // add pageInfo { hasNextPage hasPreviousPage}
                    resolvePath(node, parentType,
                            ImmutableList.<String>builder()
                                    .add("pageInfo")
                                    .add("hasNextPage").build(),
                            context);
                    resolvePath(node, parentType,
                            ImmutableList.<String>builder()
                                    .add("pageInfo")
                                    .add("hasPreviousPage").build(),
                            context);
                }
                return;
            }

            // skip numeric indexes
            if(StringUtils.isNumeric(segment) || segment.equals(Connection.FIRST) || segment.equals(Connection.LAST)) {
                resolvePath(node, parentType, path.subList(1, path.size()), context);
                return;
            }

            // auto-skip connection.edges.node
            if(getField(parentType, "node").isPresent()) {
                resolvePath(node, parentType, ImmutableList.<String>builder().add("node").addAll(path).build(), context);
                if (getField(parentType, "cursor").isPresent()) {
                    node.getChildren().put("cursor", new ResponseTreeNode());
                }
                return;
            }

            // handle attributes.byName - just grab all the attributes if set
            if(segment.equals("byName")) {
                node.children.put("name", new ResponseTreeNode());
                node.children.put("type", new ResponseTreeNode());
                node.children.put("value", new ResponseTreeNode());
                return;
            }
        }

        if(!fieldLookup.isPresent()) {
            throw new RuntimeException("bad path segment: " + segment);
        }

        Optional<IntrospectionQuery.Type> typeLookup = getResponseTypeForField(context, fieldLookup.get());
        if(!typeLookup.isPresent()) {
            throw new RuntimeException("bad type for field: " + path);
        }

        IntrospectionQuery.Type type = typeLookup.get();

        ResponseTreeNode nextNode = node.getChildren().get(segment);
        if(nextNode == null) {
            nextNode = new ResponseTreeNode();
        }

        if(path.size() > 1) {
            resolvePath(nextNode, type, path.subList(1, path.size()), context);
        }

        node.getChildren().put(segment, nextNode);
    }

    @lombok.Data
    @NoArgsConstructor
    private static final class ResponseTreeNode {

        Map<String, ResponseTreeNode> children = new HashMap<>();

        private String toQueryFragment(final List<String> path, final Context context, final List<GraphQLIntrospectionUtils.CollectedQueryParameter> parameters) {
            if(children.isEmpty()) return "";

            String childString = children.entrySet().stream()
                    .map(e -> {
                        if (StringUtils.equals(e.getKey(), "edges") ||
                                StringUtils.equals(e.getKey(), "node") ||
                                StringUtils.equals(e.getKey(), "pageInfo")) {
                            return e.getKey() + e.getValue().toQueryFragment(path, context, parameters);
                        }
                        final List<String> newPath = ImmutableList.<String>builder().addAll(path).add(e.getKey()).build();
                        return toSubQuery(e.getKey(), path,  context, parameters) + e.getValue().toQueryFragment(newPath, context, parameters);
                    })
                    .collect(Collectors.joining(" "));

            return String.format(" { __typename %s }", childString);
        }

        private String toSubQuery(final String nextSegment, final List<String> path, final Context context, final List<GraphQLIntrospectionUtils.CollectedQueryParameter> parameters) {
            if (context == null || path == null || parameters == null) {
                return nextSegment;
            }
            final String stringPath =  String.join(".", path);
            final String fullSubQueryPath = StringUtils.isEmpty(stringPath) ? nextSegment : stringPath + "." + nextSegment;
            final List<QueryParameter> queryArgs = getArgsForQuery(context, fullSubQueryPath);
            if (queryArgs == null) {
                return nextSegment;
            }
            List<String> filledQueryArgs = new ArrayList<>();
            queryArgs.forEach(arg -> {
                final String parameterNamePath = fullSubQueryPath + "." + arg.getName();
                final List<IntrospectionQuery.InputField> fields = getFieldsForInput(context, arg.getType());
                if (fields != null && !fields.isEmpty()) {
                    // Input objects, e.g. deliverBefore: DateRange
                    List<String> objectFields = new ArrayList<>();
                    fields.forEach(field -> {
                        String pathWithinInputObject = parameterNamePath + "." + field.name();
                        parameters.stream()
                                .filter(collectedQueryParameter -> StringUtils.equals(collectedQueryParameter.getPath(), pathWithinInputObject))
                                .findFirst()
                                .ifPresent(collectedQueryParameter ->
                                        objectFields.add(String.format(ARG_VAL_PATTERN, field.name(), collectedQueryParameter.getAlias())));
                    });
                    if (!objectFields.isEmpty()) {
                        filledQueryArgs.add(String.format(ARG_OBJECT_VAL_PATTERN, arg.getName(), String.join(", ", objectFields)));
                    }
                } else {
                    // Scalars: Int, String, Float etc.
                    parameters.stream()
                            .filter(collectedQueryParameter -> StringUtils.equals(collectedQueryParameter.getPath(), parameterNamePath))
                            .findFirst().ifPresent(collectedQueryParameter ->
                                    filledQueryArgs.add(String.format(ARG_VAL_PATTERN, arg.getName(), collectedQueryParameter.getAlias())));
                }
            });
            if (filledQueryArgs.isEmpty()) {
                return nextSegment;
            }
            return String.format("%s(%s)", nextSegment, String.join(", ", filledQueryArgs));
        }
    }

    private QueryDynamicVariables resolveVariablesForQuery(String query, List<CollectedQueryParameter> collectedQueryParams, Context context) {

        Map<String, Object> derivedVariables = new HashMap<>();

        // derive dynamicVariables from provided queryParameters
        collectedQueryParams.forEach(collectedQueryParameter -> {
            derivedVariables.put(collectedQueryParameter.getAlias(), collectedQueryParameter.getValue());
        });
        // check what fields are present on this entity query
        List<QueryParameter> fields = getArgsForQuery(context, query);
        // derive mandatory dynamicVariables from context
        fields.stream()
                .filter((p) -> {
                    // either argument provided manually in collectedQueryParams or it has to be taken automatically from query definition
                    String argPath = query + "." + p.getName();
                    boolean isArgumentInQueryParams = collectedQueryParams.stream()
                            .anyMatch(collectedQueryParameter -> StringUtils.equals(argPath, collectedQueryParameter.getPath()));
                    return (p.isRequired() || p.getName().equals("id")) && !isArgumentInQueryParams;
                })
                .map(p -> mapParameter(p, context))
                .forEach(e -> derivedVariables.put(e.getKey(), e.getValue()));

        return new QueryDynamicVariables(ImmutableMap.copyOf(derivedVariables));
    }

    // TODO merge with mutation version?
    private Map.Entry<String, Object> mapParameter(QueryParameter param, Context context) {
        String name = param.getName();
        Event event = context.getEvent();
        Entity entity = context.getEntity();

        switch(name) {
            case "ref": return new AbstractMap.SimpleEntry<>(name, entity.getRef());
            case "id": return new AbstractMap.SimpleEntry<>(name, entity.getId());
            case "retailer": return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of("id", event.getRetailerId()));
            case "catalogue": return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of("ref", event.getRootEntityRef()));
            case "returnOrder":
                return new AbstractMap.SimpleEntry<>(name, ImmutableMap.of(
                        "ref", event.getRootEntityRef(),
                        "retailer", ImmutableMap.of("id", event.getRetailerId())
                ));
            default: throw new RuntimeException("Cannot build a dynamic query with parameter: " + name);
        }
    }

    @Override
    public String queryDocument() {
        return fullQueryString;
    }

    @Override
    public QueryDynamicVariables variables() {
        return dynamicVariables;
    }

    @Override
    public ResponseFieldMapper<QueryDynamicData> responseFieldMapper() {
        return readerOuter -> {
            try {
                // TODO: remove double marshalling caused by dependency hell
                ObjectNode node = (ObjectNode) JsonUtils.stringToNode(
                        readerOuter.readCustomType(ResponseField.forCustomType(RESPONSE_ALIAS, RESPONSE_ALIAS, null, false, CustomType.JSON, null))
                                .toString()
                );
                return new QueryDynamicData(node);
            } catch(Exception e) {
                log.error("Exception marshalling DynamicEntityQuery response", e);
                throw new RuntimeException("Exception marshalling DynamicEntityQuery response");
            }
        };
    }

    @Override
    public QueryDynamicData wrapData(QueryDynamicData data) {
        return data;
    }

    @Nonnull
    @Override
    public OperationName name() {
        return name;
    }

    @Nonnull
    @Override
    public String operationId() {
        return UUID.randomUUID().toString();
    }
}

