package com.fluentcommerce.util.dynamic.graphql;

import com.fasterxml.jackson.databind.JavaType;
import com.fluentcommerce.util.dynamic.JsonUtils;
import com.fluentcommerce.util.dynamic.types.Connection;
import com.fluentretail.rubix.v2.context.Context;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static com.fluentcommerce.util.dynamic.graphql.GraphQLIntrospectionUtils.deriveFieldsFromJSON;

/**
 * DynamicConnectionQuery represents a qraphql loading connection fields like OrderConnection, FulfilmentConnection etc.
 * It provides clear {@link Iterable#iterator()} interface to iterate whole connection with no matter how many items it has,
 * pagination is handled internally. In general, it is enough to specify connection item class <T>, path to it in current context.
 * By default, it loads {@link Connection#DEFAULT_PAGE_COUNT} pages with {@link Connection#DEFAULT_PAGE_SIZE}.
 *
 * More specific configuration is supported {@link #DynamicConnectionQuery(Context, String, String, List, Class, Map, int, int)}  DynamicConnectionQuery}
 *
 */
public class DynamicConnectionQuery<T> implements Iterable<T> {
    private final List<Connection<T>> pages = new LinkedList<>();

    private final Context context;

    private final String pathToConnection;

    private final String queryName;
    private final List<String> fieldsPaths;
    private final Class<T> type;
    private final Map<String, Object> params;
    private final int pageLimit;

    /**
     *
     * Create an instance of DynamicConnectionQuery. It loads all fields from responseType Pojo by pathToConnection
     * e.g. class Fulfilment {id ref}.
     *
     * @param context - rule context.
     * @param queryName - name of graphqlQuery to execute.
     * @param pathToConnection inside a query response entity.
     * @param responseType of connection item. For "fulfilments" it has to be Fulfilment.class.
     * @param params query parameters to insert into final graphql query.
     */
    public DynamicConnectionQuery(Context context, String queryName,  String pathToConnection,
                                  Class<T> responseType, Map<String, Object> params) {
        this(context,
                queryName,
                pathToConnection,
                deriveFieldsFromJSON(JsonUtils.getSchemaForClass(responseType), pathToConnection),
                responseType,
                params,
                Connection.DEFAULT_PAGE_COUNT,
                Connection.DEFAULT_PAGE_SIZE);
    }


    /**
     *
     * Create an instance of DynamicConnectionQuery. It loads fields by fieldPaths for connection (pathToConnection)
     *
     * @param context - rule context.
     * @param queryName - name of graphqlQuery to execute.
     * @param pathToConnection inside a query response entity.
     * @param fieldsPaths are a list of fields to load, e.g. ["fulfilments.id", "fulfilments.ref"].
     * @param responseType of connection item. For "fulfilments" it has to be Fulfilment.class.
     * @param params query parameters to insert into final graphql query.
     */
    public DynamicConnectionQuery(Context context, String queryName, String pathToConnection,
                                  List<String> fieldsPaths, Class<T> responseType, Map<String, Object> params) {
        this(context, queryName, pathToConnection, fieldsPaths, responseType, params, Connection.DEFAULT_PAGE_COUNT, Connection.DEFAULT_PAGE_SIZE);
    }

    /**
     *
     * @param context - rule context.
     * @param queryName - name of graphqlQuery to execute. It is nullable then query name will be taken
     *                 from rule context via {@link DynamicEntityQuery#resolveQueryFromContext(Context context)}.
     *
     * @param pathToConnection inside a query response entity. which will be loaded.
     * @param connectionFieldsPaths are a list of fields to load, e.g. ["fulfilments.id", "fulfilments.ref"]
     * @param responseType of connection item. For "fulfilments" it has to be Fulfilment.class.
     * @param params - query parameters to insert into final graphql query.
     * @param pageLimit is max page count.
     * @param pageSize is max page size.
     */
    public DynamicConnectionQuery(Context context, String queryName,
                                  String pathToConnection,
                                  List<String> connectionFieldsPaths,
                                  Class<T> responseType,
                                  Map<String, Object> params,
                                  int pageLimit,
                                  int pageSize) {
        this.context = context;
        this.pathToConnection = pathToConnection;
        this.fieldsPaths = connectionFieldsPaths;
        this.type = responseType;
        this.queryName = queryName;
        this.pageLimit = pageLimit;
        if (params == null) {
            this.params = new HashMap<>();
        } else {
            this.params = params;
        }
        this.params.put(pathToConnection + "." + Connection.FIRST, pageSize);
        Optional<Connection<T>> firstPage = loadPage(null);
        firstPage.ifPresent(pages::add);
    }

    private Optional<Connection<T>> loadPage(String cursorAfter) {
        if (StringUtils.isNotEmpty(cursorAfter)) {
            this.params.put(pathToConnection + "." + Connection.AFTER, cursorAfter);
        }
        if(pages.size() < pageLimit) {
            JavaType type = JsonUtils.constructParametricType(Connection.class, this.type);
            Connection<T> page = ((DynamicDataTypes.QueryDynamicData) context.api()
                    .query(new DynamicEntityQuery(context, fieldsPaths, queryName, params)))
                    .as(type, pathToConnection);
            return Optional.ofNullable(page);
        }
        return Optional.empty();
    }

    @Override
    public ListIterator<T> iterator() {
        return new DynamicConnectionQueryIterator();
    }

    public Stream<T> stream() {
        return StreamSupport.stream(this.spliterator(), false);
    }

    private final class DynamicConnectionQueryIterator implements ListIterator<T> {
        private int page = 0;
        private int index = 0;

        @Override
        public boolean hasNext() {
            if (pages.isEmpty()) {
                return Boolean.FALSE;
            }
            Connection<T> thisPage = pages.get(page);
            if (isEmptyPage(thisPage)) {
                return false;
            }
            return index < thisPage.getEdges().size() || thisPage.getPageInfo().isHasNextPage();
        }

        @Override
        public T next() {
            if (pages.isEmpty()) {
                return null;
            }
            Connection<T> thisPage = pages.get(page);
            if (isEmptyPage(thisPage)) {
                return null;
            }
            if (index < thisPage.getEdges().size()) {
                // current page has more items
                return thisPage.getEdges().get(index++).getNode();
            } else if (page + 1 < pages.size()) {
                // another page has already been loaded
                index = 0;
                Connection<T> newPage = pages.get(page);
                return newPage.getEdges().get(index).getNode();
            } else if (thisPage.getPageInfo().isHasNextPage()) {
                // another page exists but we need to load it
                String lastCursor = thisPage.getEdges().get(thisPage.getEdges().size() - 1).getCursor();
                Optional<Connection<T>> newPage = loadPage(lastCursor);
                if (newPage.isPresent()) {
                    index = 0;
                    page++;
                    pages.add(newPage.get());
                    return newPage.get().getEdges().get(index++).getNode();
                }
            }
            return null;
        }

        @Override
        public boolean hasPrevious() {
            if (pages.isEmpty()) {
                return Boolean.FALSE;
            }
            if (index - 1 < 0) {
                 if (page - 1 >= 0) {
                     Connection<T> previousPage = pages.get(page - 1);
                     return !isEmptyPage(previousPage);
                } else {
                     return Boolean.FALSE;
                 }
            }
            return Boolean.TRUE;
        }

        @Override
        public T previous() {
            if (pages.isEmpty()) {
                return null;
            }
            if (index <= 0) {
                if (page - 1 < 0) {
                    // no previous page
                    return null;
                }
                Connection<T> previousPage = pages.get(page - 1);
                if (isEmptyPage(previousPage)) {
                    return null;
                }
                page--;
                index = previousPage.getEdges().size() - 1;
                return previousPage.getEdges().get(index).getNode();
            } else {
                index--;
                // get previous item
                return pages.get(page).getEdges().get(index).getNode();
            }

        }

        @Override
        public int nextIndex() {
            throw new UnsupportedOperationException("nextIndex");

        }

        @Override
        public int previousIndex() {
            throw new UnsupportedOperationException("previousIndex");
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("remove");
        }

        @Override
        public void set(T t) {
            throw new UnsupportedOperationException("set");
        }

        @Override
        public void add(T t) {
            throw new UnsupportedOperationException("add");
        }

        private boolean isEmptyPage(Connection<T> page) {
            return page == null  || page.getPageInfo() == null || page.getEdges() == null || page.getEdges().isEmpty();
        }
    }
}
