package com.fluentcommerce.util.dynamic;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fluentcommerce.util.dynamic.graphql.DynamicConnectionQuery;
import com.fluentcommerce.util.dynamic.graphql.DynamicDataTypes;
import com.fluentcommerce.util.dynamic.graphql.DynamicEntityQuery;
import com.fluentcommerce.util.dynamic.graphql.DynamicUpdateMutation;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Map;

public class DynamicUtils {

    /**
     * Execute a `DynamicEntityQuery` on the entity this rule is operating on.
     *
     * The structure of the query will be derived from the fields on the given
     * responseType class, i.e. if the class has a `status` field the query will
     * return the entity status. Only plain value, object fields will be loaded,
     * arrays (connections) have to be handled by {@link #queryList(Context, String, Class, Map)}  QueryList}
     *
     * @param context Rubix execution context
     * @param responseType a pojo class representing the expected response structure
     * @param <T> type of the response class
     * @return the GraphQL query response mapped to the responseType class.
     */
    public static <T> T query(Context context, Class<T> responseType) {
        return ((DynamicDataTypes.QueryDynamicData) context.api()
                .query(new DynamicEntityQuery(context, responseType))).as(responseType);
    }

    /**
     * Execute a `DynamicEntityQuery` on the entity this rule is operating on with manually propagated query parameters.
     *
     * @param context Rubix execution context
     * @param responseType a pojo class representing the expected response structure
     * @param queryParams  query parameters to insert into final graphql query.
     * @return the GraphQL query response mapped to the responseType class.
     * @param <T> type of the response class
     */
    public static <T> T query(Context context, Class<T> responseType, Map<String, Object> queryParams) {
        return ((DynamicDataTypes.QueryDynamicData) context.api()
                .query(new DynamicEntityQuery(context, responseType, queryParams))).as(responseType);
    }


    /**
     *
     * Load all connection items, graphql query name is taken from rule context, pagination is handled internally.
     * Examples (in Order context):
     *  1) load all fulfilments and its items with filledQuantity:1 and rejectedQuantity:0
     *    - context = rule context
     *    - pathToConnection = "fulfilments"
     *    - responseType = Fulfilment.class
     *    - parameters = { "fulfilments.items": { "filledQuantity": 1, "rejectedQuantity": 0 }  }
     *
     *  2) load all order items with price 100
     *     - context = rule context
     *     - pathToConnection = "items"
     *     - responseType = OrderItem.class
     *     - parameters = { "items": { "price": 100  }
     *
     *  3) load all orderItems with quantity 10 which are related to fulfilmentChoices  with dispatchOn: 2030-01-01T08:46:37.652Z and its
     *
     *     - context = rule context
     *     - pathToConnection = "fulfilmentChoices.items"
     *     - responseType = OrderItem.class
     *     - parameters = { "fulfilmentChoices": { "dispatchOn": { "from": "2030-01-01T08:46:37.652Z", "to": "2030-01-01T08:46:37.652Z" } }, "fulfilmentChoices.items": { "quantity": 10 } }
     *
     * @param context - rule context.
     * @param pathToConnection inside an query response entity.
     * @param responseType of connection item. For "fulfilments" it has to be Fulfilment.class.
     * @param parameters query parameters to insert into final graphql query.
     */
    public static <T> Iterable<T> queryList(Context context, String pathToConnection, Class<T> responseType, Map<String, Object> parameters) {
        return new DynamicConnectionQuery<>(context, null, pathToConnection, responseType, parameters);
    }


    /**
     *
     * Load all connection items, graphql query name need specifying manually, pagination is handled internally.
     *
     * @param context - rule context.
     * @param queryName - graphql query to execute from schema.
     * @param pathToConnection inside an query response entity.
     * @param responseType of connection item. For "fulfilments" it has to be Fulfilment.class.
     * @param parameters query parameters to insert into final graphql query.
     */

    public static <T> Iterable<T> queryList(Context context, String queryName, String pathToConnection, Class<T> responseType, Map<String, Object> parameters) {
        return new DynamicConnectionQuery<>(context, queryName, pathToConnection, responseType, parameters);
    }

    /**
     * Execute a `DynamicEntityQuery` on the entity this rule is operating on.
     *
     * The structure of the query will be derived from the list of paths provided,
     * e.g. if the paths include "status" and "fulfilmentChoice.deliveryType" then
     * the query would be "query { status fulfilmentChoice { deliveryType } }".
     *
     * @param context Rubix execution context
     * @param paths List of jsonPaths to build into a GraphQL query
     * @return the GraphQL query response as a `QueryDynamicData`
     */
    public static DynamicDataTypes.QueryDynamicData query(Context context, List<String> paths) {
        return ((DynamicDataTypes.QueryDynamicData) context.api()
                .query(new DynamicEntityQuery(context, paths)));
    }

    /**
     * Execute a `DynamicEntityMutation` on the entity this rule is operating on.
     *
     * Update properties of the entity based on the key-value pairs provided.
     *
     * @param context Rubix execution context
     * @param values key-value pairs to update
     */
    public static void mutate(Context context, Map<String, Object> values) {
        context.action().mutation(new DynamicUpdateMutation(context, values));
    }

    private static final String EVENT_PREFIX = "event.";

    /**
     * Get a jsonPath value from either the event or entity.
     *
     * If the path begins with `event.` it will return a value from the event,
     * in all other cases a `DynamicEntityQuery` will be executed to retrieve
     * the value from via GraphQL.
     *
     * @param context Rubix execution context
     * @param jsonPath String path to the value
     * @return a Jackson JsonNode that can be converted into the desired type
     */
    public static JsonNode getJsonPath(Context context, String jsonPath) {
        if(jsonPath.startsWith(EVENT_PREFIX)) {
            String path = jsonPath.startsWith(EVENT_PREFIX) ? jsonPath.substring(EVENT_PREFIX.length()) : jsonPath;
            ObjectNode node = JsonUtils.objectToNode(context.getEvent());

            JsonNode value = JsonUtils.getPath(node, path);
            if(value.isNull()) {
                value = JsonUtils.getPath(node, "attributes." + path);
            }

            return value;
        } else {
            return query(context, ImmutableList.of(jsonPath)).get(jsonPath);
        }
    }
}
