package com.fluentcommerce.util.core;

import com.fluentcommerce.graphql.queries.settings.GetSettingsQuery;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import lombok.Value;
import org.apache.commons.lang3.StringUtils;
import org.javatuples.Pair;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.fluentcommerce.util.core.QueryUtils.query;

public class SettingUtils {
    private static final String WILD_CARD = "%";

    /**
     * Get a single setting from the API.
     *
     * The value of the setting will be taken from the _most specific_ context that has a
     * value. So location will take precedence over retailer, then account, and finally
     * global.
     *
     * This version ignores the location context level.
     *
     * Sample usage:
     * ```
     *   String catalogue = getSetting(context, "fc.gi.product.catalogue.default")
     *                 .as(String.class).orElse("MASTER");
     * ```
     *
     * @param context current Rubix execution context
     * @param key name of the setting to retrieve
     * @return a `Setting` object with the value of the setting
     */
    public static Setting getSetting(Context context, String key) {
        return getSettings(context, ImmutableMap.of(key, key), -1).get(key);
    }

    /**
     * Overload of `getSettings(Context context, Map<String, String> keys, int locationId)`
     * for the more common use-case that doesn't need to consider location context.
     *
     * @param context current Rubix execution context
     * @param keys    settings to retrieve. The key for each can be anything that makes sense
     *                in the calling rule and will be reflected in the response Map. The value
     *                must be the name of the setting in the database.
     * @return a Map with the same keys that were passed in and values replaced with Setting
     * objects containing the setting value.
     */
    public static Map<String, Setting> getSettings(Context context, Map<String, String> keys) {
        return getSettings(context, keys, -1);
    }

    /**
     * Retrieve a list of account settings that are under the hierarchy requested from the GraphQL API.
     * <p>
     * The value of each setting will be taken from the _most specific_ context that has a
     * value.
     * <p>
     * In case any setting has both a `value` and a `lobValue`, `value` will take precedence.
     *
     * @param context current Rubix execution context
     * @param key     base setting to retrieve. All settings under this key are returned.
     * @return a Map with the same keys that were passed in and values replaced with Setting
     * objects containing the setting value.
     */
    public static Map<String, Setting> getSettingsByNamespace(Context context, String key) {
        GetSettingsQuery.Data response = query(context, GetSettingsQuery.builder()
                .settingsNames(Collections.singletonList(key + WILD_CARD))
                .retailerId(Integer.parseInt(context.getEvent().getRetailerId()))
                .build());

        final Map<String, String> keys = Maps.newHashMap();
        response.global().edges().forEach(edge2 -> keys.put(edge2.node().name(), edge2.node().name()));
        response.account().edges().forEach(edge2 -> keys.put(edge2.node().name(), edge2.node().name()));
        response.retailer().edges().forEach(edge2 -> keys.put(edge2.node().name(), edge2.node().name()));
        response.location().edges().forEach(edge2 -> keys.put(edge2.node().name(), edge2.node().name()));

        return processSettingsResponse(keys, response);
    }

    /**
     * Retrieve a list of named account settings from the GraphQL API.
     * <p>
     * The value of each setting will be taken from the _most specific_ context that has a
     * value. So location will take precedence over retailer, then account, and finally
     * global.
     * <p>
     * In case any setting has both a `value` and a `lobValue`, `value` will take precedence.
     *
     * @param context    current Rubix execution context
     * @param keys       settings to retrieve. The key for each can be anything that makes sense
     *                   in the calling rule and will be reflected in the response Map. The value
     *                   must be the name of the setting in the database.
     * @param locationId ID of the location context for this query (optional, see override)
     * @return a Map with the same keys that were passed in and values replaced with Setting
     * objects containing the setting value.
     */
    public static Map<String, Setting> getSettings(Context context, Map<String, String> keys, int locationId) {
        GetSettingsQuery.Data response = query(context, GetSettingsQuery.builder()
                .settingsNames(ImmutableList.copyOf(keys.values()))
                .retailerId(Integer.parseInt(context.getEvent().getRetailerId()))
                .locationId(locationId)
                .build());

        return processSettingsResponse(keys, response);
    }

    private static Map<String, Setting> processSettingsResponse(final Map<String, String> keys, final GetSettingsQuery.Data response) {
        return keys.entrySet().stream()
                .map((e) -> {
                    Object value = null;
                    String id = null;
                    Optional<GetSettingsQuery.Edge> loc = response.location().edges().stream()
                            .filter(s -> e.getValue().equals(s.node().name())).findFirst();
                    if (loc.isPresent()) {
                        value = StringUtils.isNotBlank(loc.get().node().value())
                                ? loc.get().node().value() : loc.get().node().lobValue();
                        id = loc.get().node().id();
                    }
                    if (value == null) {
                        Optional<GetSettingsQuery.Edge1> ret = response.retailer().edges().stream()
                                .filter(s -> e.getValue().equals(s.node().name())).findFirst();
                        if (ret.isPresent()) {
                            value = StringUtils.isNotBlank(ret.get().node().value())
                                    ? ret.get().node().value() : ret.get().node().lobValue();
                            id = ret.get().node().id();
                        }
                    }
                    if (value == null) {
                        Optional<GetSettingsQuery.Edge2> acc = response.account().edges().stream()
                                .filter(s -> e.getValue().equals(s.node().name())).findFirst();
                        if (acc.isPresent()) {
                            value = StringUtils.isNotBlank(acc.get().node().value())
                                    ? acc.get().node().value() : acc.get().node().lobValue();
                            id = acc.get().node().id();
                        }
                    }
                    if (value == null) {
                        Optional<GetSettingsQuery.Edge3> global = response.global().edges().stream()
                                .filter(s -> e.getValue().equals(s.node().name())).findFirst();
                        if (global.isPresent()) {
                            value = StringUtils.isNotBlank(global.get().node().value())
                                    ? global.get().node().value() : global.get().node().lobValue();
                            id = global.get().node().id();
                        }
                    }
                    return Pair.with(e.getKey(), new Setting(id, e.getValue(), value));
                })
                .collect(Collectors.toMap(Pair::getValue0, Pair::getValue1));
    }

    @Value
    public static class Setting {
        String id;
        String name;
        Object value;

        public <T> Optional<T> as(Class<T> type) {
            return (value == null)
                    ? Optional.empty()
                    : Optional.of(JsonUtils.convertToPojo(value, type));
        }
    }
}
