package com.fluentcommerce.util.core;

import com.fasterxml.jackson.databind.JsonNode;
import com.fluentretail.rubix.exceptions.PropertyNotFoundException;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;
import lombok.extern.slf4j.Slf4j;

import java.util.Arrays;
import java.util.List;

@Slf4j
public class RuleUtils {
    private RuleUtils() {}
    /**
     * Retrieve a rule prop as a (typed) list.
     *
     * This method is designed to support several common formats for structuring
     * list-based props as there's currently no official "standard". These include:
     * - array-literal (in the workflow JSON) of either strings or objects
     * - String of comma-separated values, e.g. "a, b, c"
     *
     * @param context current Rubix execution context
     * @param name String name of the prop to convert
     * @param clazz literal reference to T
     * @param <T> expected type of the items in the response List
     * @return List<T> of the prop values configured for this rule
     */
    public static <T> List<T> rulePropAsList(Context context, String name, Class<T> clazz) {
        Object value = context.getProp(name, Object.class);
        ImmutableList.Builder<Object> builder = ImmutableList.builder();
        if (value instanceof List) {
            builder.addAll((List) value);
        } else if (value.getClass().isArray()) {
            builder.addAll(ImmutableList.copyOf((Object[]) value));
        } else if (value instanceof String) {
            builder.addAll(
                Arrays.stream(((String) value).split(","))
                    .map(String::trim)
                    .map((str) -> {
                        JsonNode node = JsonUtils.stringToNode(str);
                        return (node == null) ? str : node;
                    })
                    .collect(new CollectionUtils.ImmutableListCollector<>())
            );
        } else {
            builder.add(value);
        }
        return builder.build().stream()
                .map((obj) -> JsonUtils.convertToPojo(obj, clazz))
                .collect(new CollectionUtils.ImmutableListCollector<T>());
    }

    /**
     * Given the context of a rubix execution, and a list of rule props, returns true if all props are not empty.
     * False otherwise.
     *
     * @param context the context of a rubix execution
     * @param ruleProps a list of rule property keys to retrieve from the context
     * @return true if the list of rule properties are present and non-empty in the context.
     */
    public static boolean validateRulePropsIsNotEmpty(Context context, String... ruleProps) {
        if (ruleProps != null) {
            for (String ruleProp : ruleProps) {
                if (null == context.getProp(ruleProp) || context.getProp(ruleProp).isEmpty()) {
                    log.error("The event property " + ruleProp + " is expected but was not present on " + context.getEvent().getName());
                    throw new PropertyNotFoundException(400, ruleProp);
                }
            }
        }
        return true;
    }
}
