package com.fluentcommerce.util.core;

import com.apollographql.apollo.api.Operation;
import com.apollographql.apollo.api.Query;
import com.fluentretail.rubix.exceptions.RubixException;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableList;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.function.Function;

public class QueryUtils {

    /**
     * Properly generified helper version of `context.api().query()`
     *
     * @param <T> the expected response type of the GQL query.
     * @return query result cast to T
     */
    public static <T extends Operation.Data> T query(Context context, Query<T, ?, ?> query) {
        return (T) context.api().query(query);
    }

    public static final String CONNECTION_EDGE = "edges";
    public static final String CONNECTION_NODE = "node";

    /**
     * Variant of `connectionToList` that will apply a converter function (from T to R) and return a list
     * of results instead of whatever was originally in the collection.
     *
     * @param connection Apollo data object implementing a connection-like naming scheme
     * @param converter function to convert each item in the connection from T to R
     * @param <T> type of object in the connection
     * @param <R> type the objects will be converted to
     * @return ImmutableList list of Node values
     */
    public static <T, R> ImmutableList<R> connectionToList(Object connection, Function<T, R> converter) {
        ImmutableList<T> list =  connectionToList(connection, false);
        ImmutableList.Builder<R> builder = ImmutableList.builder();
        list.forEach(node -> {
            R r = converter.apply(node);
            if(r != null) builder.add(r);
        });
        return builder.build();
    }


    /**
     * Overload of `connectionToList` with `exceptionOnFail` defaulted to `false`
     *
     * @param connection Apollo data object implementing a connection-like naming scheme
     * @return ImmutableList list of Node values
     */
    public static <T> ImmutableList<T> connectionToList(Object connection) {
        return connectionToList(connection, false);
    }

    /**
     * Null-safe helper method to turn an Apollo-generated Connection instance into a List of things.
     *
     * For example, if a query for an Order returns an ItemConnection, this would turn that straight into a list of
     * Item instances without having to write multiple levels of null checks and iterators.
     *
     * This method requires that the aliases on the GQL query are named or suffixed with "edge" and "node" respectively.
     * Note that if you have multiple Connections within a single query Apollo automatically names them Node1, Node2 etc
     * which won't work, so best to alias them as ItemNode and FulfilmentNode etc (and better for code readability too!)
     *
     * @param connection Apollo data object implementing a connection-like naming scheme
     * @param exceptionOnFail if true, throw an exception if the connection is not valid.
     *                        Will return an empty list otherwise.
     * @param <T> the data type of the Node value within the connection
     * @return ImmutableList list of Node values
     */
    public static <T> ImmutableList<T> connectionToList(Object connection, boolean exceptionOnFail) {
        // TODO: would it be quicker just to serialise/deserialise this via Jackson?
        if(connection == null) return ImmutableList.of();

        try {
            Method edgeMethod = null;
            for(Method m : connection.getClass().getMethods()) {
                if(m.getName().toLowerCase().endsWith(CONNECTION_EDGE)) {
                    edgeMethod = m; break;
                }
            }
            if(edgeMethod == null) throw new NoSuchMethodException();

            List<Object> edges = (List<Object>) edgeMethod.invoke(connection);
            if(edges != null && edges.size() > 0) {
                Object first = edges.get(0);
                Method nodeMethod = null;
                for(Method m : first.getClass().getMethods()) {
                    if(m.getName().toLowerCase().endsWith(CONNECTION_NODE)) {
                        nodeMethod = m; break;
                    }
                }
                if(nodeMethod == null) throw new NoSuchMethodException();

                ImmutableList.Builder<T> builder = ImmutableList.builder();
                for(Object o : edges) {
                    builder.add((T) nodeMethod.invoke(o));
                }
                return builder.build();
            }
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            if(exceptionOnFail)
                throw new RubixException(500, "Object does not appear to be a valid connection object: " + connection);
        }
        return ImmutableList.of();
    }
}
